<?php

namespace Ignite\Vendor\Helix\Laravel\Http\Responses;

use Illuminate\Contracts\Support\Responsable;
use Illuminate\Http\JsonResponse;

class ApiResponse implements Responsable
{
    const STATUS_ERROR = 'error';
    const STATUS_SUCCESS = 'success';

    /**
     * @var int
     */
    protected $code = JsonResponse::HTTP_OK;

    /**
     * @var string
     */
    protected $status = self::STATUS_SUCCESS;

    /**
     * @var string
     */
    protected $message = '';

    /**
     * @var array
     */
    protected $data = [];

    /**
     * @var array
     */
    protected $headers = [];

    /**
     * @var int
     */
    protected $options = 0;

    /**
     * Make a new instance of an API Response.
     *
     * @return ApiResponse
     */
    public static function make()
    {
        return new static();
    }

    /**
     * Mark the response as a success.
     *
     * @return $this
     */
    public function success()
    {
        $this->code(JsonResponse::HTTP_OK);
        $this->status = static::STATUS_SUCCESS;

        return $this;
    }

    /**
     * Mark the response as a failure.
     *
     * @param int $code
     *
     * @return $this
     */
    public function error($code = JsonResponse::HTTP_SERVICE_UNAVAILABLE)
    {
        $this->code($code);
        $this->status = static::STATUS_ERROR;

        return $this;
    }

    /**
     * The status code.
     *
     * @param int $code
     *
     * @return $this
     */
    public function code(int $code)
    {
        $this->code = $code;

        return $this;
    }

    /**
     * The status message.
     *
     * @param string $message
     *
     * @return $this
     */
    public function message(string $message)
    {
        $this->message = $message;

        return $this;
    }

    /**
     * Set the merge data.
     *
     * @param string $name
     * @param string $value
     *
     * @return $this
     */
    public function header(string $name, string $value)
    {
        $this->headers[$name] = $value;

        return $this;
    }

    /**
     * The data formatted correctly for Helix.
     *
     * @param array $data
     *
     * @return $this
     */
    public function data(array $data = [])
    {
        $this->data = $data;

        return $this;
    }

    /**
     * Create an HTTP response that represents the object.
     *
     * @param \Illuminate\Http\Request $request
     *
     * @return JsonResponse
     */
    public function toResponse($request)
    {
        return new JsonResponse(array_merge($this->data, [
            'status' => $this->status,
            'code' => $this->code,
            'message' => $this->message
        ]), $this->code, $this->headers, $this->options);
    }
}
