<?php

namespace Ignite\Vendor\Helix\Laravel\Resources\Concerns;

use Illuminate\Database\Eloquent\Builder;

/**
 * Statusable trait
 *
 * @property string $status
 */
trait Statusable
{
    /**
     * Boot the Statusable trait when used on an Eloquent model.
     *
     * @return void
     */
    public static function bootStatusable()
    {
        static::saving(function ($model) {
            /** @var Statusable $model */
            if ($model->wasProcessing() && $model->isPurging()) {
                $model->guardAgainstStatusChangeFromProcessingToPurging();
            }
            if ($model->wasPurging() && $model->isProcessing()) {
                $model->guardAgainstStatusChangeFromPurgingToProcessing();
            }
        });

        static::deleting(function ($model) {
            /** @var Statusable $model */
            if ($model->wasProcessing() || $model->isProcessing()) {
                $model->guardAgainstDeleteProcessing();
            }
        });
    }

    /**
     * Determine whether the webhook is processing.
     *
     * @return bool
     */
    public function isProcessing()
    {
        return $this->status === static::STATUS_PROCESSING;
    }

    /**
     * Determine whether the webhook was processing but has been changed.
     *
     * @return bool
     */
    public function wasProcessing()
    {
        return $this->getOriginal('status') === static::STATUS_PROCESSING;
    }

    /**
     * Scope a query to include resources with a status of `processing`.
     *
     * @param Builder $query
     *
     * @return void
     */
    public function scopeWithProcessing(Builder $query)
    {
        $query->where('status', static::STATUS_PROCESSING);
    }

    /**
     * Determine whether the webhook is processing.
     *
     * @return bool
     */
    public function isProcessed()
    {
        return $this->status === static::STATUS_PROCESSED;
    }

    /**
     * Determine whether the webhook was processed but has been changed.
     *
     * @return bool
     */
    public function wasProcessed()
    {
        return $this->getOriginal('status') === static::STATUS_PROCESSED;
    }

    /**
     * Scope a query to include resources with a status of `processed`.
     *
     * @param Builder $query
     *
     * @return void
     */
    public function scopeWithProcessed(Builder $query)
    {
        $query->where('status', static::STATUS_PROCESSED);
    }

    /**
     * Determine whether the webhook is purging, about to be purged.
     *
     * @return bool
     */
    public function isPurging()
    {
        return $this->status === static::STATUS_PURGING;
    }

    /**
     * Determine whether the webhook was purging but has been changed.
     *
     * @return bool
     */
    public function wasPurging()
    {
        return $this->getOriginal('status') === static::STATUS_PURGING;
    }

    /**
     * Scope a query to include resources with a status of `purging`.
     *
     * @param Builder $query
     *
     * @return void
     */
    public function scopeWithPurging(Builder $query)
    {
        $query->where('status', static::STATUS_PURGING);
    }

    /**
     * Set the status to processing.
     *
     * @return $this
     */
    public function processing()
    {
        $this->update(['status' => static::STATUS_PROCESSING]);

        return $this->fresh();
    }

    /**
     * Set the status to processed.
     *
     * @return $this
     */
    public function processed()
    {
        $this->update(['status' => static::STATUS_PROCESSED]);

        return $this->fresh();
    }

    /**
     * Set the status to processed.
     *
     * @return $this
     * @throws \Exception
     */
    public function purging()
    {
        if ($this->isProcessing()) {
            $this->guardAgainstStatusChangeFromProcessingToPurging();
        }

        $this->update(['status' => static::STATUS_PURGING]);

        return $this->fresh();
    }

    /**
     * Guard against a status being changed from `processing` to `purging`.
     *
     * @throws \Exception
     */
    public function guardAgainstStatusChangeFromProcessingToPurging()
    {
        throw new \Exception('You cannot purge a webhook with a status of `processing`.');
    }

    /**
     * Guard against a status being changed from `purging` to `processing`.
     *
     * @throws \Exception
     */
    public function guardAgainstStatusChangeFromPurgingToProcessing()
    {
        throw new \Exception('You cannot set a `purging` webhook back to `processing`.');
    }

    /**
     * Guard against a webhook being deleted when it's `status` is `processing`.
     *
     * @throws \Exception
     */
    public function guardAgainstDeleteProcessing()
    {
        throw new \Exception('You cannot delete a webhook with a status of `processing`.');
    }
}
