<?php

namespace Ignite\Vendor\Helix\Laravel\Listeners;

use Ignite\Vendor\Helix\Laravel\Contracts\DecoratableSsoClientInterface;
use Ignite\Vendor\Helix\Laravel\Exceptions\PayloadValidationFailed;
use Ignite\Vendor\Helix\Laravel\Exceptions\SsoBadLoginUrl;
use Ignite\Vendor\Helix\Laravel\Exceptions\SsoRequestFailed;
use Ignite\Vendor\Helix\Laravel\Listeners\Concerns\GuardAgainstFailedSsoRequest;
use Ignite\Vendor\Helix\Laravel\Listeners\Concerns\GuardAgainstMissingLoginUrl;
use Ignite\Vendor\Helix\Laravel\Listeners\Concerns\RetrievesSsoSessionKey;
use Ignite\Vendor\Helix\Laravel\Response;
use Ignite\Vendor\Helix\Laravel\Sso\Client;
use Ignite\Vendor\Helix\Laravel\Sso\Payload;
use Illuminate\Auth\Events\Login;

class SyncOnLogin
{
    use GuardAgainstFailedSsoRequest, GuardAgainstMissingLoginUrl, RetrievesSsoSessionKey;

    /**
     * @var Client
     */
    private $client;

    /**
     * Create the event listener.
     *
     * @param DecoratableSsoClientInterface $client
     */
    public function __construct(DecoratableSsoClientInterface $client)
    {
        $this->client = $client;
    }

    /**
     * Handle the event.
     *
     * @param Login $event
     *
     * @return void
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function handle(Login $event)
    {
        $sessionKey = $this->getSessionKey();

        try {
            /** @var Payload $payload */
            $payload = $event->user->payload();

            /** @var Response $response */
            $response = $this->client->signIn($payload);

            $this->guardAgainstFailedSsoRequest($payload, $response);
            $this->guardAgainstMissingLoginUrl($response);

            request()->session()->put($sessionKey, $response->get(Response::KEY_SSO_URL));
        } catch (PayloadValidationFailed $exception) {
            $errors = $exception->getErrors();
            logger()->error($exception->getMessage(), compact('exception', 'errors', 'event'));

            request()->session()->put($sessionKey, Client::SSO_INVALID_DATA);
        } catch (SsoRequestFailed $exception) {
            $payload = $exception->getPayload();
            $response = $exception->getResponse();
            logger()->error($exception->getMessage(), compact('exception', 'response', 'payload', 'event'));

            request()->session()->put($sessionKey, Client::SSO_INVALID_RESPONSE);
        } catch (SsoBadLoginUrl $exception) {
            $response = $exception->getResponse();
            logger()->error($exception->getMessage(), compact('exception', 'response', 'event'));

            request()->session()->put($sessionKey, Client::SSO_INVALID_LOGIN_URL);
        } catch (\Exception $exception) {
            logger()->error($exception->getMessage(), compact('exception', 'event'));

            request()->session()->put($sessionKey, Client::SSO_UNKNOWN_ERROR);
        }
    }
}
