<?php

namespace Ignite\Vendor\Helix\Laravel;

use Ignite\Vendor\Helix\Laravel\Commands\DecryptAccessToken;
use Ignite\Vendor\Helix\Laravel\Commands\EncryptAccessToken;
use Ignite\Vendor\Helix\Factory;
use Ignite\Vendor\Helix\HelixInterface;
use Ignite\Vendor\Helix\Laravel\Commands\PurgeProcessedResources;
use Ignite\Vendor\Helix\Laravel\Services\Token;
use Illuminate\Foundation\Application;
use Illuminate\Support\ServiceProvider;

class HelixServiceProvider extends ServiceProvider
{
    /**
     * When Laravel is booting up, register configuration.
     */
    public function boot()
    {
        $this->loadRoutesFrom(__DIR__ . '/../routes/api.php');

        $this->loadMigrationsFrom(__DIR__.'/../migrations');

        $this->publishes([
            $this->helixConfigPath() => config_path('helix.php'),
        ]);

        if ($this->app->runningInConsole()) {
            $this->commands([
                DecryptAccessToken::class,
                EncryptAccessToken::class,
                PurgeProcessedResources::class,
            ]);
        }
    }

    /**
     * When Laravel is registering service providers, register the ClientInterface
     * and provide an alias for use with a Facade.
     */
    public function register()
    {
        $this->mergeConfigFrom($this->helixConfigPath(), 'helix');

        $this->app->singleton(Token::class, $this->closure('registerTokenService'));

        $this->app->singleton(HelixInterface::class, $this->closure('registerHelix'));

        $this->app->alias(HelixInterface::class, 'helix');
    }

    /**
     * Register the Helix client customized for Laravel.
     *
     * @param Application $app
     *
     * @return \Ignite\Vendor\Helix\DecoratorInterface
     */
    public function registerHelix($app)
    {
        $config = $app['config']->get('helix');

        $factory = new Factory($config['adapters'], $config['decorators']);

        return $factory->make(
            $config['username'],
            $config['password'],
            $config['endpoint'],
            $config['adapter'],
            $config['decorator']
        );
    }

    /**
     * Register the Access Token Encryption and Decryption Service.
     *
     * @param Application $app
     *
     * @return Token
     * @throws \Exception
     */
    public function registerTokenService($app)
    {
        return new Token(
            $app['config']->get('helix.token.key'),
            $app['config']->get('helix.token.cipher')
        );
    }

    /**
     * Create a Closure from a callable.
     *
     * @param string $method
     *
     * @return \Closure
     */
    protected function closure(string $method)
    {
        return \Closure::fromCallable([$this, $method]);
    }

    /**
     * The path to the config file.
     *
     * @return bool|string
     */
    protected function helixConfigPath()
    {
        return realpath(__DIR__ . '/../config/helix.php');
    }
}
