<?php

namespace Ignite\Vendor\Helix\Laravel;

use Illuminate\Support\Collection;
use Illuminate\Support\Fluent;

class Response
{
    const STATUS_SUCCESS = 'success';
    const STATUS_FAILURE = 'error';

    /**
     * @var array
     */
    protected $attributes;

    /**
     * @var Collection
     */
    protected $collection;

    /**
     * Response constructor.
     *
     * @param array $attributes
     */
    public function __construct($attributes = [])
    {
        $this->attributes = $attributes;
    }

    /**
     * The status of the request.
     *
     * @return string
     */
    public function status()
    {
        return $this->get('status', self::STATUS_FAILURE);
    }

    /**
     * Determine if the status of the request failed.
     *
     * @return string
     */
    public function message()
    {
        return $this->get('message');
    }

    /**
     * The data as a collection of fluent objects.
     *
     * @return Collection
     */
    public function data()
    {
        if (! $this->collection) {
            $this->collection = (new Collection($this->attributes['data'] ?? []))
                ->mapInto(Fluent::class);
        }

        return $this->collection;
    }

    /**
     * Determine if the status of the request passed.
     *
     * @return string
     */
    public function passed()
    {
        return $this->get('status') === self::STATUS_SUCCESS;
    }

    /**
     * Determine if the status of the request failed.
     *
     * @return string
     */
    public function failed()
    {
        return $this->get('status') === self::STATUS_FAILURE;
    }

    /**
     * Get the value of the given key.
     *
     * @param  string $key
     * @param  mixed $default
     * @return mixed
     */
    public function get($key, $default = null)
    {
        if (! array_key_exists($key, $this->attributes)) {
            return $default;
        }

        return $this->attributes[$key];
    }

    /**
     * Proxy unrecognized methods to the data collection.
     *
     * @param string $name
     * @param array $paramaters
     * @return mixed
     * @throws \Exception
     */
    public function __call($name, $paramaters)
    {
        if (! method_exists($this->data(), $name)) {
            throw new \Exception("Unknown method {$name}().");
        }
        return call_user_func_array([$this->data(), $name], $paramaters);
    }
}
