<?php

namespace Ignite\Vendor\Hawk\Tests;

use Ignite\Vendor\Hawk\PersonalizedRecipient;
use PHPUnit\Framework\TestCase;

class PersonalizedRecipientTest extends TestCase
{
    /** @var array */
    protected $addressDataFixture = [
        'line1' => '102 Decker Ct',
        'city' => 'Irving',
        'region' => 'TX',
        'postalCode' => '75062',
        'country' => 'USA',
    ];

    /**
     * @test
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     */
    public function it_can_create_an_instance_from_an_associative_array()
    {
        $this->assertInstanceOf(PersonalizedRecipient::class, PersonalizedRecipient::fromAssociativeArray([
            'firstName' => 'John',
            'lastName' => 'Smith',
            'address' => $this->addressDataFixture,
        ]));
    }

    /**
     * @test
     * @expectedException \InvalidArgumentException
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     */
    public function it_fails_when_first_name_is_missing()
    {
        PersonalizedRecipient::fromAssociativeArray([
            'firstName' => '',
            'lastName' => 'Smith',
            'address' => $this->addressDataFixture,
        ]);
    }

    /**
     * @test
     * @expectedException \InvalidArgumentException
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     */
    public function it_fails_when_first_name_exceeds_character_limit()
    {
        PersonalizedRecipient::fromAssociativeArray([
            'firstName' => str_repeat('X', 26),
            'lastName' => 'Smith',
            'address' => $this->addressDataFixture,
        ]);
    }

    /**
     * @test
     * @expectedException \InvalidArgumentException
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     */
    public function it_fails_when_last_name_is_missing()
    {
        PersonalizedRecipient::fromAssociativeArray([
            'firstName' => 'John',
            'lastName' => '',
            'address' => $this->addressDataFixture,
        ]);
    }

    /**
     * @test
     * @expectedException \InvalidArgumentException
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     */
    public function it_fails_when_last_name_exceeds_character_limit()
    {
        PersonalizedRecipient::fromAssociativeArray([
            'firstName' => 'John',
            'lastName' => str_repeat('X', 26),
            'address' => $this->addressDataFixture,
        ]);
    }

    /**
     * @test
     * @expectedException \InvalidArgumentException
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     */
    public function it_fails_when_address_is_an_empty_array()
    {
        PersonalizedRecipient::fromAssociativeArray([
            'firstName' => 'John',
            'lastName' => 'Smith',
            'address' => [],
        ]);
    }

    /**
     * @test
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     */
    public function it_converts_an_array_of_address_details_into_a_recipient_address_instance_and_casts_back_to_array()
    {
        $recipient = PersonalizedRecipient::fromAssociativeArray([
            'firstName' => 'John',
            'lastName' => 'Smith',
            'address' => $this->addressDataFixture,
        ]);

        $this->assertInternalType('array', $recipient->toArray()['address']);
        $this->assertArrayHasKey('line1', $recipient->toArray()['address']);
    }

    /**
     * @test
     * @expectedException \InvalidArgumentException
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     */
    public function it_fails_when_id_exceeds_character_limit()
    {
        PersonalizedRecipient::fromAssociativeArray([
            'firstName' => 'John',
            'lastName' => 'Smith',
            'address' => $this->addressDataFixture,
            'id' => str_repeat('X', 31),
        ]);
    }

    /**
     * @test
     * @expectedException \InvalidArgumentException
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     */
    public function it_fails_when_email_exceeds_character_limit()
    {
        PersonalizedRecipient::fromAssociativeArray([
            'firstName' => 'John',
            'lastName' => 'Smith',
            'address' => $this->addressDataFixture,
            'email' => str_repeat('X', 51)
        ]);
    }

    /**
     * @test
     * @expectedException \InvalidArgumentException
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     */
    public function it_fails_when_email_is_invalid()
    {
        PersonalizedRecipient::fromAssociativeArray([
            'firstName' => 'John',
            'lastName' => 'Smith',
            'address' => $this->addressDataFixture,
            'email' => 'not_a_real_email_address'
        ]);
    }

    /**
     * @test
     * @expectedException \InvalidArgumentException
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     */
    public function it_fails_when_prefix_exceeds_character_limit()
    {
        PersonalizedRecipient::fromAssociativeArray([
            'firstName' => 'John',
            'lastName' => 'Smith',
            'address' => $this->addressDataFixture,
            'prefix' => str_repeat('X', 6)
        ]);
    }

    /**
     * @test
     * @expectedException \InvalidArgumentException
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     */
    public function it_fails_when_middlename_exceeds_character_limit()
    {
        PersonalizedRecipient::fromAssociativeArray([
            'firstName' => 'John',
            'lastName' => 'Smith',
            'address' => $this->addressDataFixture,
            'middleName' => str_repeat('X', 26)
        ]);
    }

    /**
     * @test
     * @expectedException \InvalidArgumentException
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     */
    public function it_fails_when_suffix_exceeds_character_limit()
    {
        PersonalizedRecipient::fromAssociativeArray([
            'firstName' => 'John',
            'lastName' => 'Smith',
            'address' => $this->addressDataFixture,
            'suffix' => str_repeat('X', 6)
        ]);
    }

    /**
     * @test
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     */
    public function it_can_get_the_recipient_canonical_name()
    {
        $recipient = PersonalizedRecipient::fromAssociativeArray([
            'firstName' => 'John',
            'lastName' => 'Smith',
            'address' => $this->addressDataFixture
        ]);

        $this->assertSame('John Smith', $recipient->getCanonicalName());
    }

    /**
     * @test
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     */
    public function it_can_get_the_recipient_full_name()
    {
        $recipient = PersonalizedRecipient::fromAssociativeArray([
            'prefix' => 'Mr',
            'firstName' => 'John',
            'middleName' => 'Matthew',
            'lastName' => 'Smith',
            'suffix' => 'Snr',
            'address' => $this->addressDataFixture,
        ]);

        $this->assertSame('Mr John Matthew Smith Snr', $recipient->getFullName());
    }

    /**
     * @test
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     */
    public function it_includes_non_required_params_when_available()
    {
        $recipient = PersonalizedRecipient::fromAssociativeArray([
            'prefix' => 'Mr',
            'firstName' => 'John',
            'middleName' => 'Matthew',
            'lastName' => 'Smith',
            'suffix' => 'Snr',
            'address' => $this->addressDataFixture,
            'phoneNumber' => '555-123-4567',
            'id' => 'ID1'
        ]);

        $this->assertArrayHasKey('id', $recipient->toArray());
        $this->assertArrayHasKey('phoneNumber', $recipient->toArray());
        $this->assertArrayHasKey('prefix', $recipient->toArray());
        $this->assertArrayHasKey('middleName', $recipient->toArray());
        $this->assertArrayHasKey('suffix', $recipient->toArray());
    }

    /**
     * @test
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     */
    public function it_does_not_includes_non_required_params_when_not_available()
    {
        $recipient = PersonalizedRecipient::fromAssociativeArray([
            'prefix' => null,
            'firstName' => 'John',
            'middleName' => null,
            'lastName' => 'Smith',
            'suffix' => '',
            'address' => $this->addressDataFixture
        ]);

        $this->assertArrayNotHasKey('id', $recipient->toArray());
        $this->assertArrayNotHasKey('phoneNumber', $recipient->toArray());
        $this->assertArrayNotHasKey('prefix', $recipient->toArray());
        $this->assertArrayNotHasKey('middleName', $recipient->toArray());
        $this->assertArrayNotHasKey('suffix', $recipient->toArray());
    }
}
