<?php

namespace Ignite\Vendor\Hawk;

class ThirdPartyInfo extends Model
{
    /**
     * @var array
     */
    protected $address;

    /**
     * @var null|string
     */
    protected $taxId;

    /**
     * Create an instance from an associative array, assuming that array has the required keys.
     *
     * @param  array $data
     * @return $this
     */
    public static function fromAssociativeArray($data)
    {
        return app(static::class, $data);
    }

    /**
     * Constructor
     *
     * @param null|string $company
     * @param null|string $contact
     * @param null|string $line1
     * @param null|string $line2
     * @param null|string $line3
     * @param null|string $city
     * @param null|string $region
     * @param null|string $postalCode
     * @param null|string $postalCodeExt
     * @param null|string $country
     * @param null|string $taxId
     */
    public function __construct(
        ?string $company = null,
        ?string $contact = null,
        ?string $line1 = null,
        ?string $line2 = null,
        ?string $line3 = null,
        ?string $city = null,
        ?string $region = null,
        ?string $postalCode = null,
        ?string $postalCodeExt = null,
        ?string $country = null,
        ?string $taxId = null
    ) {
        $this->address = [];
        $this->setCompany($company);
        $this->setContact($contact);
        $this->setLine1($line1);
        $this->setLine2($line2);
        $this->setLine3($line3);
        $this->setCity($city);
        $this->setRegion($region);
        $this->setPostalCode($postalCode);
        $this->setPostalCodeExt($postalCodeExt);
        $this->setCountry($country);

        $this->setTaxId($taxId);
    }

    /**
     * Get the instance as an array.
     *
     * @return array
     */
    public function toArray()
    {
        return array_filter(parent::toArray());
    }

    /**
     * Guard clause to disallow invalid regions.
     *
     * @param string $name
     * @param string $value
     */
    protected function guardAgainstInvalidRegion($name, $value)
    {
        // Address region must be a valid 2-character uppercase abbreviation for a U.S. state or a Canadian province
        // or contain an international region less than 36 characters.
        $threshold = in_array($this->address['country'] ?? '', ['USA', 'US', 'CAN', 'CA']) ? 2 : 35;

        $this->guardAgainstMaxLength($name, $value, $threshold);
    }

    /**
     * Sets the address company.
     *
     * @param string|null $company
     * @return self
     */
    protected function setCompany(?string $company = null): self
    {
        $this->guardAgainstMaxLength('company', $company, 60);

        $this->address['company'] = $company;

        return $this;
    }

    /**
     * Sets the address contact.
     *
     * @param string|null $contact
     * @return self
     */
    protected function setContact(?string $contact = null): self
    {
        $this->guardAgainstMaxLength('contact', $contact, 35);

        $this->address['contact'] = $contact;

        return $this;
    }

    /**
     * Sets the address line1.
     *
     * @param string|null $line1
     * @return self
     */
    protected function setLine1(?string $line1 = null): self
    {
        $this->guardAgainstMaxLength('line1', $line1, 35);

        $this->address['line1'] = $line1;

        return $this;
    }

    /**
     * Sets the address line2.
     *
     * @param string|null $line2
     * @return self
     */
    protected function setLine2(?string $line2 = null): self
    {
        $this->guardAgainstMaxLength('line2', $line2, 35);

        $this->address['line2'] = $line2;

        return $this;
    }

    /**
     * Sets the address line3.
     *
     * @param string|null $line3
     * @return self
     */
    protected function setLine3(?string $line3 = null): self
    {
        $this->guardAgainstMaxLength('line3', $line3, 35);

        $this->address['line3'] = $line3;

        return $this;
    }

    /**
     * Sets the address city.
     *
     * @param string|null $city
     * @return self
     */
    protected function setCity(?string $city = null): self
    {
        $this->guardAgainstMinLength('city', $city, 1);
        $this->guardAgainstMaxLength('city', $city, 30);

        $this->address['city'] = $city;

        return $this;
    }

    /**
     * Sets the address region.
     *
     * @param string|null $region
     * @return self
     */
    protected function setRegion(?string $region = null): self
    {
        $this->guardAgainstMinLength('region', $region, 1);

        $this->guardAgainstInvalidRegion('region', $region);

        $this->address['region'] = $region;

        return $this;
    }

    /**
     * Sets the address postalCode.
     *
     * @param string|null $postalCode
     * @return self
     */
    protected function setPostalCode(?string $postalCode = null): self
    {
        $this->guardAgainstMinLength('postalCode', $postalCode, 1);
        $this->guardAgainstMaxLength('postalCode', $postalCode, 10);

        $this->address['postalCode'] = $postalCode;

        return $this;
    }

    /**
     * Sets the address postalCodeExt.
     *
     * @param string|null $postalCodeExt
     * @return self
     */
    protected function setPostalCodeExt(?string $postalCodeExt = null): self
    {
        $this->guardAgainstMaxLength('postalCodeExt', $postalCodeExt, 4);

        $this->address['postalCodeExt'] = $postalCodeExt;

        return $this;
    }

    /**
     * Sets the address country.
     *
     * @param string|null $country
     * @return self
     */
    protected function setCountry(?string $country = null): self
    {
        // The third party's country must be the 3 character alphabetic ISO code
        // corresponding to the country.

        $this->guardAgainstMinLength('country', $country, 3);
        $this->guardAgainstMaxLength('country', $country, 3);

        // now that country is set, check region again
        $this->guardAgainstInvalidRegion('region', $this->address['region'] ?? '');

        $this->address['country'] = $country;

        return $this;
    }

    /**
     * Sets the taxId.
     *
     * @param string|null $taxId
     * @return self
     */
    protected function setTaxId(?string $taxId = null): self
    {
        $this->guardAgainstMinLength('taxId', $taxId, 1);
        $this->guardAgainstMaxLength('taxId', $taxId, 9);

        if (!is_numeric($taxId)) {
            throw new \InvalidArgumentException("Invalid taxId, must be numeric: {$taxId}");
        }

        $this->taxId = $taxId;

        return $this;
    }
}
