<?php

namespace Ignite\Vendor\Hawk;

use Illuminate\Contracts\Support\Arrayable;

abstract class Model implements Arrayable
{
    /**
     * Get the instance as an array.
     *
     * @return array
     */
    public function toArray()
    {
        return array_filter(array_map(function ($value) {
            return $value instanceof Arrayable ? $value->toArray() : $value;
        }, get_object_vars($this)));
    }

    /**
     * Call to get a property dynamically given the property actually exists.
     *
     * @param  string $method
     * @param  array  $arguments
     * @return mixed
     */
    public function __call($method, $arguments)
    {
        $action = strtolower(substr($method, 0, 3));

        if ('get' !== $action && 'has' !== $action) {
            throw new \DomainException('Hawk models are immutable. You must use e.g. `get{PROPERTY}()` or `has{PROPERTY}()`.');
        }

        $property = $this->camelCase(substr($method, 3));

        if (! property_exists($this, $property)) {
            return null;
        }

        if ('has' === $action) {
            return $this->getBooleanValue($property);
        }

        return $this->$property;
    }

    /**
     * Determine true or false based on the type.
     *
     * @param string $property
     * @return bool
     */
    private function getBooleanValue($property)
    {
        $value = $this->$property;

        if (is_bool($value) || is_null($value)) {
            return (bool) $value;
        }

        if (is_array($value)) {
            return ! empty(array_filter($value));
        }

        if (is_string($value)) {
            $value = trim($value);
            return ! empty($value);
        }

        if (is_object($value) && method_exists($value, 'isEmpty')) {
            return $value->isEmpty();
        }
    }

    /**
     * Guard clause to disallow values over a certain threshold.
     *
     * @param string $name
     * @param string $value
     * @param int    $threshold
     */
    protected function guardAgainstMaxLength($name, $value, $threshold)
    {
        if (strlen($value ?? 0) > $threshold) {
            throw new \InvalidArgumentException("Invalid $name, exceeds maximum length of $threshold character(s): $value");
        }
    }

    /**
     * Guard clause to disallow values over a certain threshold.
     *
     * @param string $name
     * @param string $value
     * @param int    $threshold
     */
    protected function guardAgainstMinLength($name, $value, $threshold)
    {
        if (strlen($value ?? 0) < $threshold) {
            throw new \InvalidArgumentException("Invalid $name, requires minimum length of $threshold character(s): $value");
        }
    }

    /**
     * Guard clause to disallow invalid email values.
     *
     * @param string $email
     */
    protected function guardAgainstInvalidEmail($email)
    {
        if (! filter_var($email, FILTER_VALIDATE_EMAIL)) {
            throw new \InvalidArgumentException("Invalid email, please submit a correctly formatted email address: $email");
        }
    }

    /**
     * Truncate values that are exceeds the max length allowed by Hawk's API.
     *
     * Recommended for values that are flexible and can be truncated (after a certain length) and still send "mostly"
     * the right data. Others should use guardAgainstMaxLength() where they can revise for correction.
     *
     * @param  string      $name
     * @param  string|null $value
     * @param  int         $threshold
     * @return string|null
     */
    protected function transformExceedingMaxLength(string $name, $value, int $threshold): ?string
    {
        if (strlen($value) > $threshold) {
            $trimmed = substr($value, 0, $threshold);
            logger()->debug(
                "Parameter {$name} exceeds the max length {$threshold} and will be trimmed: {$value} => {$trimmed}"
            );

            return $trimmed;
        }

        return $value;
    }

    /**
     * Camel case the given string.
     *
     * @param string $value
     *
     * @return string
     */
    protected function camelCase($value)
    {
        return str_replace(' ', '', lcfirst(ucwords(str_replace(['-', '_'], ' ', $value))));
    }
}
