<?php

namespace Ignite\Vendor\Hawk;

use Illuminate\Contracts\Support\Arrayable;

class ClosedLoopIndividualOrderLine extends Model implements Arrayable
{
    /**
     * @var PersonalizedRecipient
     */
    protected $recipient;

    /**
     * @var string
     */
    protected $amount;

    /**
     * @var null|string
     */
    protected $contentProvider;

    /**
     * @var null|string
     */
    protected $groupingId;

    /**
     * @var null|string
     */
    protected $description;

    /**
     * @var string
     */
    protected $shippingMethod;

    /**
     * @var null|string
     */
    protected $cardCarrierMessage;

    /**
     * @var null|string
     */
    protected $fourthLineEmbossText;

    /**
     * @var array
     */
    protected $additionalInfo;

    /**
     * Create an instance from an associative array, assuming that array has the required keys.
     *
     * @param  array $data
     * @return static
     */
    public static function fromAssociativeArray(array $data)
    {
        return app(static::class, $data);
    }

    /**
     * ClosedLoopIndividualOrderLine constructor.
     *
     * @param array|PersonalizedRecipient $recipient
     * @param string                      $amount
     * @param string                      $contentProvider
     * @param string                      $description
     * @param string                      $groupingId
     * @param string                      $shippingMethod
     * @param string                      $cardCarrierMessage
     * @param string                      $fourthLineEmbossText
     * @param array                       $additionalInfo
     */
    public function __construct(
        $recipient,
        $amount,
        $contentProvider = null,
        $description = null,
        $groupingId = null,
        $shippingMethod = ShippingMethod::USPS_FIRST_CLASS,
        $cardCarrierMessage = null,
        $fourthLineEmbossText = null,
        array $additionalInfo = []
    ) {
        $this->setRecipient($recipient);
        $this->setAmount($amount);
        $this->setContentProvider($contentProvider);
        $this->setGroupingId($groupingId);
        $this->setDescription($description);
        $this->setShippingMethod($shippingMethod);
        $this->setCardCarrierMessage($cardCarrierMessage);
        $this->setFourthLineEmbossText($fourthLineEmbossText);
        $this->additionalInfo = $additionalInfo;
    }

    /**
     * Set the recipient.
     *
     * @param  array|PersonalizedRecipient $recipient
     * @return $this
     */
    private function setRecipient($recipient)
    {
        if (is_array($recipient) && ! empty($recipient)) {
            $recipient = app(PersonalizedRecipient::class, $recipient);
        }

        if (! $recipient instanceof PersonalizedRecipient) {
            throw new \InvalidArgumentException("Invalid recipient, you must pass an instance of PersonalizedRecipient or an array that can be converted.");
        }

        $this->recipient = $recipient;

        return $this;
    }

    /**
     * Set the amount.
     *
     * @param  mixed $amount
     * @return $this
     */
    private function setAmount($amount)
    {
        if (! is_numeric($amount)) {
            throw new \InvalidArgumentException("Invalid amount, must be numeric: $amount");
        }

        $this->amount = (float) $amount;

        return $this;
    }

    /**
     * Set the content provider.
     *
     * @param  string $contentProvider
     * @return $this
     */
    private function setContentProvider($contentProvider)
    {
        $this->contentProvider = $contentProvider;

        return $this;
    }

    /**
     * Set the fourth line emboss text.
     *
     * @param  string $groupingId
     * @return $this
     */
    private function setGroupingId($groupingId)
    {
        if (! is_null($groupingId)) {
            $this->guardAgainstMaxLength('groupingId', $groupingId, 50);
        }

        $this->groupingId = $groupingId;

        return $this;
    }

    /**
     * Set the fourth line emboss text.
     *
     * @param  string $description
     * @return $this
     */
    private function setDescription($description)
    {
        if (! is_null($description)) {
            $this->guardAgainstMaxLength('description', $description, 40);
        }

        $this->description = $description;

        return $this;
    }

    /**
     * Set the shipping method.
     *
     * @param  string $shippingMethod
     * @return $this
     */
    private function setShippingMethod($shippingMethod)
    {
        if (! ShippingMethod::accepts($shippingMethod)) {
            throw new \InvalidArgumentException("Unrecognized shipping method $shippingMethod.");
        }

        $this->shippingMethod = $shippingMethod;

        return $this;
    }

    /**
     * Set the fourth line emboss text.
     *
     * @param  string $cardCarrierMessage
     * @return $this
     */
    private function setCardCarrierMessage($cardCarrierMessage)
    {
        if (! is_null($cardCarrierMessage)) {
            $this->guardAgainstMaxLength('cardCarrierMessage', $cardCarrierMessage, 80);
        }

        $this->cardCarrierMessage = $cardCarrierMessage;

        return $this;
    }

    /**
     * Set the fourth line emboss text.
     *
     * @param  string $fourthLineEmbossText
     * @return $this
     */
    private function setFourthLineEmbossText($fourthLineEmbossText)
    {
        if (! is_null($fourthLineEmbossText)) {
            $this->guardAgainstMaxLength('fourthLineEmbossText', $fourthLineEmbossText, 21);
        }

        $this->fourthLineEmbossText = $fourthLineEmbossText;

        return $this;
    }
}
