<?php

namespace Ignite\Vendor\Hawk\Tests;

use GuzzleHttp\Exception\GuzzleException;
use Ignite\Vendor\Hawk\ClosedLoopBulkOrderLine;
use Ignite\Vendor\Hawk\ClosedLoopIndividualOrderLine;
use Ignite\Vendor\Hawk\EgiftIndividualOrderLine;
use Ignite\Vendor\Hawk\EmailContent;
use Ignite\Vendor\Hawk\FundingOrderLine;
use Ignite\Vendor\Hawk\PaymentType;
use Ignite\Vendor\Hawk\PersonalizedOrderLine;
use Ignite\Vendor\Hawk\Response\SubmitVirtualIndividualResponse;
use Ignite\Vendor\Hawk\Response\SubmitVirtualIndividualTransactionResponse;
use Ignite\Vendor\Hawk\VirtualIndividualOrderLine;
use Ignite\Vendor\Hawk\Request\SubmitClosedLoopBulkRequest;
use Ignite\Vendor\Hawk\Request\SubmitClosedLoopIndividualRequest;
use Ignite\Vendor\Hawk\Request\SubmitEgiftIndividualRequest;
use Ignite\Vendor\Hawk\Request\SubmitFundingRequest;
use Ignite\Vendor\Hawk\Request\SubmitOpenLoopPersonalizedIndividualRequest;
use Ignite\Vendor\Hawk\Request\SubmitVirtualIndividualRequest;
use Ignite\Vendor\Hawk\Response\OrderInfoResponse;
use Ignite\Vendor\Hawk\Response\SubmitClosedLoopBulkResponse;
use Ignite\Vendor\Hawk\Response\SubmitClosedLoopIndividualResponse;
use Ignite\Vendor\Hawk\Response\SubmitEGiftIndividualResponse;
use Ignite\Vendor\Hawk\Response\SubmitFundingResponse;
use Ignite\Vendor\Hawk\Response\SubmitOpenLoopPersonalizedIndividualResponse;
use Ignite\Vendor\Hawk\ShippingMethod;
use Ignite\Vendor\Hawk\Ssl;
use Ignite\Vendor\Hawk\UnsubscribeData;
use Ignite\Vendor\Hawk\VirtualRecipient;
use Ignite\Vendor\Hawk\Client;
use PHPUnit\Framework\TestCase;

class ClientTest extends TestCase
{
    const UNIVERSAL_VISA = '43476908';
    const CLOSED_LOOP_US_PHYSICAL_BULK = '43507608';
    const CLOSED_LOOP_US_PHYSICAL_INDIVIDUAL = '43507508';
    const CLOSED_LOOP_US_EGIFT_INDIVIDUAL = '43479508';
    const OPEN_LOOP_PERSONALIZED_INDIVIDUAL = '43513208';
    const VIRTUAL_INDIVIDUAL_MASTERCARD = '43477008';

    /** @var Client */
    protected $client;

    /**
     * Get the test configured instance of the Hawk client.
     *
     * @return Client
     */
    private function getClientInstance()
    {
        if (is_null($this->client)) {
            $this->client = new Client(env('API_MERCHANT'), $this->getSslInstance(), null, true);
        }

        return $this->client;
    }

    /**
     * Get the test configured instance the Ssl object.
     *
     * @return Ssl
     */
    private function getSslInstance()
    {
        return new Ssl(
            realpath(__DIR__ . '/../' . env('API_CERT')),
            realpath(__DIR__ . '/../' . env('API_KEY'))
        );
    }

    /**
     * @test
     * @throws \Ignite\Vendor\Hawk\ResponseException
     * @throws GuzzleException
     */
    public function it_can_get_catalog_product_info()
    {
        $clientProgramId = '43479508';
        //$clientProgramId = self::CLOSED_LOOP_US_EGIFT_INDIVIDUAL;
        $requestId = 'TEST_REQUEST_ID_' . date('Y_m_d_H_i_s');
        $response = $this->getClientInstance()->clientProgram($clientProgramId, $requestId);

        dd($response);
        $this->assertIsInt($response->getClientProgramId());
        $this->assertEquals($clientProgramId, $response->getClientProgramId());
        $this->assertIsString($response->getProgramName());
        $this->assertEquals('BrightSpot - US eGift ACH DEBIT - TEST Quote', $response->getProgramName());
        $this->assertIsString($response->getProgramType());
        $this->assertEquals('DIGITAL', $response->getProgramType());
        $this->assertIsString($response->getCurrency());
        $this->assertEquals('USD', $response->getCurrency());
        $this->assertIsArray($response->getProducts());
    }

    /**
     * @test
     * @group Api
     * @group Vendor
     * @group OpenLoop
     * @group ApiCommunication
     * @throws \GuzzleHttp\Exception\GuzzleException
     * @throws \Ignite\Vendor\Hawk\ResponseException
     */
    public function it_can_send_an_egift_individual_order_request()
    {
        $requestId = 'TEST_REQUEST_ID_' . date('Y_m_d_H_i_s');
        $requestPo = 'TEST_PO_NUMBER_' . date('Y_m_d_H_i_s');
        $product = self::CLOSED_LOOP_US_EGIFT_INDIVIDUAL;

        $request = SubmitEGiftIndividualRequest::fromAssociativeArray([
            'orderDetails' => [
                EgiftIndividualOrderLine::fromAssociativeArray([
                    'recipient' => VirtualRecipient::fromAssociativeArray([
                        'firstName' => 'John',
                        'lastName' => 'Smith',
                        'email' => 'johnsmith@gmail.com',
                        //'id' => '12345',
                    ]),
                    'amount' => 10,
                    'contentProvider' => "AmericanEagleOutfitters_eGift",
                ])
            ],
            'clientProgramNumber' => $product,
            'paymentType' => PaymentType::DRAW_DOWN,
            'poNumber' => $requestPo,
            'returnCardNumberAndPIN' => true,
            'emailContent' => EmailContent::fromAssociativeArray([
                'senderName' => 'Ignite Platform',
                'senderEmail' => 'it@gobrightspot.com',
                'subject' => 'Test eGift Subject',
                'emailMessage' => 'Test eGift Message',
                'emailClosingName' => 'Test eGift Closing Name',
                'unsubscribeData' => UnsubscribeData::fromAssociativeArray([
                    'companyName' => 'Brightspot',
                    'addressLine1' => '102 Decker Court',
                    'addressCity' => 'Irving',
                    'addressState' => 'TX',
                    'addressZip' => '75062',
                    'addressCountry' => 'US',
                    'methodType' => 'EMAIL',
                    'value' => 'it@brightspot.email',
                ])
            ]),
        ]);

        $response = $this->getClientInstance()->submitEgiftIndividual($request, $requestId);

        $this->assertInstanceOf(SubmitEGiftIndividualResponse::class, $response);
    }

    /**
     * @test
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     * @group ApiCommunication
     * @throws \GuzzleHttp\Exception\GuzzleException
     * @throws \Ignite\Vendor\Hawk\ResponseException
     */
    public function it_can_send_a_closed_loop_individual_order_request()
    {
        $requestId = 'TEST_REQUEST_ID_' . date('Y_m_d_H_i_s');
        $requestPo = 'TEST_PO_NUMBER_' . date('Y_m_d_H_i_s');
        $product = self::CLOSED_LOOP_US_PHYSICAL_INDIVIDUAL;

        $request = SubmitClosedLoopIndividualRequest::fromAssociativeArray([
            'orderDetails' => [
                ClosedLoopIndividualOrderLine::fromAssociativeArray([
                    'recipient' => [
                        'firstName' => 'John',
                        'lastName' => 'Smith',
                        'address' => [
                            'line1' => '102 Decker Ct',
                            'city' => 'Irving',
                            'region' => 'TX',
                            'postalCode' => '75062',
                            'country' => "USA",
                            'line2' => 'Suite 150'
                        ],
                    ],
                    'amount' => 100,
                    'quantity' => 5,
                    'contentProvider' => "Amazoncom",
                    'description' => 'The description text',
                    'groupingId' => '1',
                    'shippingMethod' => ShippingMethod::USPS_FIRST_CLASS,
                    'cardCarrierMessage' => null,
                    'fourthLineEmbossText' => null,
                ])
            ],
            'clientProgramNumber' => $product,
            'paymentType' => PaymentType::ACH_DEBIT,
            'poNumber' => $requestPo,
        ]);

        $response = $this->getClientInstance()->submitClosedLoopIndividual($request, $requestId);

        $this->assertInstanceOf(SubmitClosedLoopIndividualResponse::class, $response);
    }

    /**
     * @test
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group ClosedLoop
     * @group ApiCommunication
     * @throws \GuzzleHttp\Exception\GuzzleException
     * @throws \Ignite\Vendor\Hawk\ResponseException
     */
    public function it_can_send_a_closed_loop_bulk_order_request()
    {
        $requestId = 'TEST_REQUEST_ID_' . date('Y_m_d_H_i_s');
        $requestPo = 'TEST_PO_NUMBER_' . date('Y_m_d_H_i_s');
        $product = self::CLOSED_LOOP_US_PHYSICAL_BULK;

        $request = SubmitClosedLoopBulkRequest::fromAssociativeArray([
            'orderDetails' => [
                ClosedLoopBulkOrderLine::fromAssociativeArray([
                    'recipient' => [
                        'firstName' => 'John',
                        'lastName' => 'Smith',
                        'address' => [
                            'line1' => '102 Decker Ct',
                            'city' => 'Irving',
                            'region' => 'TX',
                            'postalCode' => '75062',
                            'country' => "USA",
                            'line2' => 'Suite 150'
                        ],
                    ],
                    'deliveryAddress' => [
                        'company' => 'Brightspot',
                        'line1' => '102 Decker Ct',
                        'city' => 'Irving',
                        'region' => 'TX',
                        'postalCode' => '75062',
                        'country' => 'USA',
                        'line2' => 'Suite 150',
                        'line3' => '',
                    ],
                    'amount' => 100,
                    'quantity' => 5,
                    'contentProvider' => "Amazoncom",
                    'description' => 'The description text',
                    'groupingId' => '1',
                    'shippingMethod' => ShippingMethod::USPS_FIRST_CLASS,
                    'cardCarrierMessage' => null,
                    'fourthLineEmbossText' => null,
                ])
            ],
            'clientProgramNumber' => $product,
            'paymentType' => PaymentType::DRAW_DOWN,
            'poNumber' => $requestPo,
        ]);

        $response = $this->getClientInstance()->submitClosedLoopBulk($request, $requestId);

        $this->assertInstanceOf(SubmitClosedLoopBulkResponse::class, $response);
    }

    /**
     * @test
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group Reloadable
     * @group ApiCommunication
     * @throws \GuzzleHttp\Exception\GuzzleException
     * @throws \Ignite\Vendor\Hawk\ResponseException
     */
    public function it_can_send_an_open_loop_personalized_individual_order_request()
    {
        $requestId = 'TEST_REQUEST_ID_' . date('Y_m_d_H_i_s');
        $requestPo = 'TEST_PO_NUMBER_' . date('Y_m_d_H_i_s');
        $product = self::OPEN_LOOP_PERSONALIZED_INDIVIDUAL;

        $request = SubmitOpenLoopPersonalizedIndividualRequest::fromAssociativeArray([
            'orderDetails' => [
                PersonalizedOrderLine::fromAssociativeArray([
                    'recipient' => [
                        'id' => uniqid('TEST'),
                        'firstName' => 'John',
                        'lastName' => 'Smith',
                        'address' => [
                            'line1' => '102 Decker Ct',
                            'city' => 'Irving',
                            'region' => 'TX',
                            'postalCode' => '75062',
                            'country' => "USA",
                            'line2' => 'Suite 150'
                        ],
                    ],
                    'amount' => 100,
                    'quantity' => 1,
                    'description' => 'The description text',
                    'shippingMethod' => ShippingMethod::USPS_FIRST_CLASS,
                    'cardCarrierMessage' => null,
                    'fourthLineEmbossText' => null,
                ])
            ],
            'clientProgramNumber' => $product,
            'paymentType' => PaymentType::DRAW_DOWN,
            'poNumber' => $requestPo,
        ]);

        $response = $this->getClientInstance()->submitOpenLoopPersonalizedIndividual($request, $requestId);

        $this->assertInstanceOf(SubmitOpenLoopPersonalizedIndividualResponse::class, $response);
    }

    /**
     * @test
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group Reloadable
     * @group ApiCommunication
     * @throws \Ignite\Vendor\Hawk\ResponseException
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function it_can_submit_funding_for_reloadable_request()
    {
        $requestId = 'TEST_REQUEST_ID_' . date('Y_m_d_H_i_s');
        $requestPo = 'TEST_PO_NUMBER_' . date('Y_m_d_H_i_s');
        $product = self::OPEN_LOOP_PERSONALIZED_INDIVIDUAL;

        $request = SubmitFundingRequest::fromAssociativeArray([
            'orderDetails' => [
                FundingOrderLine::fromAssociativeArray([
                    'recipient' => [
                        'id' => 9
                    ],
                    'amount' => 100,
                    'description' => 'The description text',
                ])
            ],
            'clientProgramNumber' => $product,
            'paymentType' => PaymentType::DRAW_DOWN,
            'poNumber' => $requestPo,
        ]);

        $response = $this->getClientInstance()->submitFunding($request, $requestId);

        $this->assertInstanceOf(SubmitFundingResponse::class, $response);
    }

    /**
     * @test
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group Reloadable
     * @group ApiCommunication
     * @throws \GuzzleHttp\Exception\GuzzleException
     * @throws \Ignite\Vendor\Hawk\ResponseException
     */
    public function it_can_send_a_virtual_individual_order_request()
    {
        $requestId = 'TEST_REQUEST_ID_' . date('Y_m_d_H_i_s');
        $requestPo = 'TEST_PO_NUMBER_' . date('Y_m_d_H_i_s');
        $product = self::VIRTUAL_INDIVIDUAL_MASTERCARD;

        $request = SubmitVirtualIndividualRequest::fromAssociativeArray([
            'orderDetails' => [
                VirtualIndividualOrderLine::fromAssociativeArray([
                    'clientRefId' => '12345',
                    'recipient' => VirtualRecipient::fromAssociativeArray([
                        'firstName' => 'John',
                        'lastName' => 'Smith',
                        'email' => 'johnsmith@gmail.com',
                        //'id' => '12345',
                    ]),
                    'amount' => 100,
                    'description' => "IG200100001234",
                ])
            ],
            'clientProgramNumber' => $product,
            'paymentType' => PaymentType::DRAW_DOWN,
            'poNumber' => $requestPo,
            'emailContent' => EmailContent::fromAssociativeArray([
                'senderName' => 'Ignite Platform',
                'senderEmail' => 'it@gobrightspot.com',
                'subject' => 'Test eGift Subject',
                'emailMessage' => 'Test eGift Message',
                'emailClosingName' => 'Test eGift Closing Name',
                'unsubscribeData' => UnsubscribeData::fromAssociativeArray([
                    'companyName' => 'Brightspot',
                    'addressLine1' => '102 Decker Court',
                    'addressCity' => 'Irving',
                    'addressState' => 'TX',
                    'addressZip' => '75062',
                    'addressCountry' => 'US',
                    'methodType' => 'EMAIL',
                    'value' => 'it@brightspot.email',
                ])
            ]),
        ]);

        $response = $this->getClientInstance()->submitVirtualIndividual($request, $requestId, 2000);

        $this->assertInstanceOf(SubmitVirtualIndividualResponse::class, $response);
    }

    /**
     * @test
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group Reloadable
     * @group ApiCommunication
     * @throws \Ignite\Vendor\Hawk\ResponseException
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function it_can_get_the_transaction_details_for_a_virtual_individual_order_request()
    {
        $requestId = 'TRANSACTION_REQUEST_ID_' . date('Y_m_d_H_i_s');
        $requestPo = 'TRANSACTION_PO_NUMBER_' . date('Y_m_d_H_i_s');
        $product = self::VIRTUAL_INDIVIDUAL_MASTERCARD;

        $request = SubmitVirtualIndividualRequest::fromAssociativeArray([
            'orderDetails' => [
                VirtualIndividualOrderLine::fromAssociativeArray([
                    //'clientRefId' => date('YmdHis') . '_' . uniqid(),
                    'recipient' => VirtualRecipient::fromAssociativeArray([
                        'firstName' => 'John',
                        'lastName' => 'Smith',
                        'email' => 'johnsmith@gmail.com',
                        //'id' => date('YmdHis') . '_' . uniqid(),
                    ]),
                    'amount' => 100,
                    'description' => "IG200100001234",
                ])
            ],
            'clientProgramNumber' => $product,
            'paymentType' => PaymentType::ACH_DEBIT,
            'poNumber' => $requestPo,
            'emailContent' => EmailContent::fromAssociativeArray([
                'senderName' => 'Ignite Platform',
                'senderEmail' => 'it@gobrightspot.com',
                'subject' => 'Test eGift Subject',
                'emailMessage' => 'Test eGift Message',
                'emailClosingName' => 'Test eGift Closing Name',
                'unsubscribeData' => UnsubscribeData::fromAssociativeArray([
                    'companyName' => 'Brightspot',
                    'addressLine1' => '102 Decker Court',
                    'addressCity' => 'Irving',
                    'addressState' => 'TX',
                    'addressZip' => '75062',
                    'addressCountry' => 'US',
                    'methodType' => 'EMAIL',
                    'value' => 'it@brightspot.email',
                ])
            ]),
        ]);

        $response = $this->getClientInstance()->submitVirtualIndividual($request, $requestId, 2000);

        dd($response);
        $this->assertInstanceOf(SubmitVirtualIndividualResponse::class, $response);

        sleep(5);

        $response = $this->getClientInstance()->submitVirtualIndividualTransaction($response->getTransactionId());

        dd($response);

        $this->assertInstanceOf(SubmitVirtualIndividualTransactionResponse::class, $response);

    }

    /**
     * @test
     * @group Catalog
     * @group Api
     * @group Vendor
     * @group Reloadable
     * @group ApiCommunication
     * @throws \Ignite\Vendor\Hawk\ResponseException
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function it_can_retrieve_order_info()
    {
        $orderNumber = 401064159;
        $orderRequestId = "TEST_REQUEST_ID_2018_09_14_19_07_39";
        $requestId = 'TEST_REQUEST_ID_' . date('Y_m_d_H_i_s');
        $product = self::OPEN_LOOP_PERSONALIZED_INDIVIDUAL;

        $response = $this->getClientInstance()->orderInfo($orderNumber, null, null, $requestId);

        $this->assertInstanceOf(OrderInfoResponse::class, $response);
    }
}
