<?php

namespace Ignite\Vendor\Hawk\Response;

use Ignite\Vendor\Hawk\Model;
use Ignite\Vendor\Hawk\ResponseException;
use Psr\Http\Message\MessageInterface;
use Psr\Http\Message\ResponseInterface;
use Psr\Http\Message\StreamInterface;

abstract class Response extends Model
{
    /**
     * @var ResponseInterface
     */
    private $response;

    /**
     * Parse the json string.
     *
     * @param  ResponseInterface $response
     *
     * @return array
     * @throws ResponseException
     */
    public static function parseJson(ResponseInterface $response): array
    {
        $data = json_decode($response->getBody(), true);

        if (JSON_ERROR_NONE !== json_last_error()) {
            throw new ResponseException($response, 'JSON parse error: ' . json_last_error_msg());
        }

        return $data;
    }

    /**
     * Set the HTTP client response.
     *
     * @param  ResponseInterface $response
     * @return $this
     */
    public function setHttpResponse(ResponseInterface $response): self
    {
        $this->response = $response;

        return $this;
    }

    /**
     * The HTTP response headers.
     *
     * @return string[][]
     */
    public function getHeaders(): array
    {
        return $this->response->getHeaders();
    }

    /**
     * A specific HTTP response header.
     *
     * @param  string $name
     * @return string[]
     */
    public function getHeader($name): array
    {
        return $this->response->getHeader($name);
    }

    /**
     * Get the HTTP client response.
     *
     * @return null|ResponseInterface
     */
    public function getHttpResponse(): ?ResponseInterface
    {
        return $this->response;
    }

    /**
     * The HTTP response status code.
     *
     * @return int
     */
    public function getStatusCode(): int
    {
        return $this->response->getStatusCode();
    }

    /**
     * The HTTP response error reason phrase.
     *
     * @return string
     */
    public function getReasonPhrase(): string
    {
        return $this->response->getReasonPhrase();
    }

    /**
     * The HTTP response body before transformation.
     *
     * @return StreamInterface
     */
    public function getBody(): StreamInterface
    {
        return $this->response->getBody();
    }

    /**
     * Retrieves the HTTP protocol version as a string.
     *
     * @return string HTTP protocol version.
     */
    public function getProtocolVersion(): string
    {
        return $this->response->getProtocolVersion();
    }

    /**
     * Return an instance with the specified HTTP protocol version.
     *
     * @param string $version HTTP protocol version
     * @return MessageInterface
     */
    public function withProtocolVersion($version): MessageInterface
    {
        return $this->response->withProtocolVersion($version);
    }

    /**
     * Checks if a header exists by the given case-insensitive name.
     *
     * @param  string $name
     * @return bool
     */
    public function hasHeader($name): bool
    {
        return $this->response->hasHeader($name);
    }

    /**
     * Retrieves a comma-separated string of the values for a single header.
     *
     * @param  string $name
     * @return string
     */
    public function getHeaderLine($name): string
    {
        return $this->response->getHeaderLine($name);
    }

    /**
     * Return an instance with the provided value replacing the specified header.
     *
     * @param string          $name
     * @param string|string[] $value
     * @return MessageInterface
     * @throws \InvalidArgumentException for invalid header names or values.
     */
    public function withHeader($name, $value): MessageInterface
    {
        return $this->response->withHeader($name, $value);
    }

    /**
     * Return an instance with the specified header appended with the given value.
     *
     * @param string          $name
     * @param string|string[] $value
     * @return MessageInterface
     * @throws \InvalidArgumentException for invalid header names or values.
     */
    public function withAddedHeader($name, $value): MessageInterface
    {
        return $this->response->withAddedHeader($name, $value);
    }

    /**
     * Return an instance without the specified header.
     *
     * @param string $name Case-insensitive header field name to remove.
     * @return MessageInterface
     */
    public function withoutHeader($name): MessageInterface
    {
        return $this->response->withoutHeader($name);
    }

    /**
     * Return an instance with the specified message body.
     *
     * @param  StreamInterface $body
     * @return MessageInterface
     * @throws \InvalidArgumentException When the body is not valid.
     */
    public function withBody(StreamInterface $body): MessageInterface
    {
        return $this->response->withBody($body);
    }

    /**
     * Return an instance with the specified status code and, optionally, reason phrase.
     *
     * @param int    $code
     * @param string $reasonPhrase
     * @return ResponseInterface
     */
    public function withStatus($code, $reasonPhrase = ''): ResponseInterface
    {
        return $this->response->withStatus($code, $reasonPhrase);
    }
}
