<?php

namespace Ignite\Vendor\Hawk;

class PersonalizedRecipient extends Model
{
    /**
     * @var string
     */
    protected $firstName;

    /**
     * @var string
     */
    protected $lastName;

    /**
     * @var RecipientAddress
     */
    protected $address;

    /**
     * @var string
     */
    protected $phoneNumber;

    /**
     * @var null|string
     */
    protected $id;

    /**
     * @var null|string
     */
    protected $email;

    /**
     * @var null|string
     */
    protected $prefix;

    /**
     * @var null|string
     */
    protected $middleName;

    /**
     * @var null|string
     */
    protected $suffix;

    /**
     * Create an instance from an associative array, assuming that array has the required keys.
     *
     * @param  array $data
     * @return $this
     */
    public static function fromAssociativeArray($data)
    {
        return app(static::class, $data);
    }

    /**
     * PersonalizedRecipient constructor.
     *
     * @param string                 $firstName
     * @param string                 $lastName
     * @param array|RecipientAddress $address
     * @param null|string            $id
     * @param string                 $phoneNumber
     * @param null|string            $email
     * @param null|string            $prefix
     * @param null|string            $middleName
     * @param null|string            $suffix
     */
    public function __construct(
        $firstName,
        $lastName,
        $address,
        $id = null,
        $phoneNumber = null,
        $email = null,
        $prefix = null,
        $middleName = null,
        $suffix = null
    ) {
        $this->setFirstName($firstName);
        $this->setLastName($lastName);
        $this->setAddress($address);
        $this->setPhoneNumber($phoneNumber);
        $this->setId($id);
        $this->setEmail($email);
        $this->setPrefix($prefix);
        $this->setMiddleName($middleName);
        $this->setSuffix($suffix);
    }

    /**
     * Set the first name.
     *
     * @param  string $firstName
     * @return $this
     */
    private function setFirstName($firstName)
    {
        $this->guardAgainstMaxLength('firstName', $firstName, 25);
        $this->guardAgainstMinLength('firstName', $firstName, 1);

        $this->firstName = $firstName;

        return $this;
    }

    /**
     * Set the last name.
     *
     * @param  string $lastName
     * @return $this
     */
    private function setLastName($lastName)
    {
        $this->guardAgainstMaxLength('lastName', $lastName, 25);
        $this->guardAgainstMinLength('lastName', $lastName, 1);

        $this->lastName = $lastName;

        return $this;
    }

    /**
     * Set the address.
     *
     * @param  array|RecipientAddress $address
     * @return $this
     */
    private function setAddress($address)
    {
        if (is_array($address) && ! empty($address)) {
            $address = app(RecipientAddress::class, $address);
        }

        if (! $address instanceof RecipientAddress) {
            throw new \InvalidArgumentException("Invalid address, you must pass an instance of RecipientAddress or an array that can be converted.");
        }

        $this->address = $address;

        return $this;
    }

    /**
     * Set the ID.
     *
     * @param  string $id
     * @return $this
     */
    private function setId($id)
    {
        if (! is_null($id)) {
            $this->guardAgainstMaxLength('id', $id, 30);
        }

        $this->id = $id;

        return $this;
    }

    /**
     * Set the email.
     *
     * @param  string $email
     * @return $this
     */
    private function setEmail($email = null)
    {
        if (! is_null($email)) {
            $this->guardAgainstMaxLength('email', $email, 50);
            $this->guardAgainstInvalidEmail($email);
        }

        $this->email = $email;

        return $this;
    }

    /**
     * Set the prefix.
     *
     * @param string $prefix
     * @return $this
     */
    private function setPrefix($prefix = null)
    {
        if (! is_null($prefix)) {
            $this->guardAgainstMaxLength('prefix', $prefix, 5);
        }

        $this->prefix = $prefix;

        return $this;
    }

    /**
     * Set the middle name.
     *
     * @param  null|string $middleName
     * @return $this
     */
    private function setMiddleName($middleName = null)
    {
        if (! is_null($middleName)) {
            $this->guardAgainstMaxLength('middleName', $middleName, 25);
        }

        $this->middleName = $middleName;

        return $this;
    }

    /**
     * Set the suffix.
     *
     * @param  null|string $suffix
     * @return $this
     */
    private function setSuffix($suffix = null)
    {
        if (! is_null($suffix)) {
            $this->guardAgainstMaxLength('suffix', $suffix, 5);
        }

        $this->suffix = $suffix;

        return $this;
    }

    /**
     * Set the phone number.
     *
     * @param null|string $phoneNumber
     */
    private function setPhoneNumber($phoneNumber = null)
    {
        if (! is_null($phoneNumber)) {
            // we decided we wanted to sanitize the phoneNumber before sending
            $phoneNumber = preg_replace('/[^0-9]/', '', $phoneNumber);
            $this->guardAgainstMinLength('phoneNumber', $phoneNumber, 10);
        }

        $this->phoneNumber = $phoneNumber;
    }

    /**
     * Get the canonical name, the name by which the recipient is "known".
     *
     * @return string
     */
    public function getCanonicalName()
    {
        return $this->firstName . ' ' . $this->lastName;
    }

    /**
     * Get the full name.
     *
     * @return string
     */
    public function getFullName()
    {
        $name = $this->firstName;

        if (! empty($this->prefix)) {
            $name = $this->prefix . ' ' . $name;
        }

        if (! empty($this->middleName)) {
            $name = $name . ' ' . $this->middleName;
        }

        $name = $name . ' ' . $this->lastName;

        if (! empty($this->suffix)) {
            $name = $name . ' ' . $this->suffix;
        }

        return $name;
    }

    /**
     * Get the instance as an array.
     *
     * @return array
     */
    public function toArray()
    {
        return array_filter(parent::toArray());
    }
}
