<?php

namespace Ignite\Vendor\Hawk;

class EmailContent extends Model
{
    /**
     * @var string
     */
    protected $senderName;

    /**
     * @var string
     */
    protected $senderEmail;

    /**
     * @var string
     */
    protected $subject;

    /**
     * @var string
     */
    protected $emailMessage;

    /**
     * @var string
     */
    protected $emailClosingName;

    /**
     * @var UnsubscribeData
     */
    protected $unsubscribeData;


    /**
     * VirtualRecipient constructor.
     *
     * @param string                $senderName
     * @param string                $senderEmail
     * @param string                $subject
     * @param string                $emailMessage
     * @param string                $emailClosingName
     * @param array|UnsubscribeData $unsubscribeData
     */
    public function __construct($senderName, $senderEmail, $subject, $emailMessage, $emailClosingName, $unsubscribeData)
    {
        $this->setSenderName($senderName);
        $this->setSenderEmail($senderEmail);
        $this->setSubject($subject);
        $this->setEmailMessage($emailMessage);
        $this->setEmailClosingName($emailClosingName);
        $this->setUnsubscribeData($unsubscribeData);
    }

    /**
     * Create a new instance from an associative array.
     *
     * @param  array $data
     * @return self
     */
    public static function fromAssociativeArray(array $data)
    {
        return app(static::class, $data);
    }

    /**
     * Set the Sender full name.
     *
     * @param  string $senderName
     * @return self
     */
    public function setSenderName($senderName)
    {
        $this->guardAgainstMaxLength('senderName', $senderName, $threshold = 50);
        $this->senderName = $senderName;

        return $this;
    }

    /**
     * Set the sender email address.
     *
     * @param  string $senderEmail
     * @return self
     */
    public function setSenderEmail($senderEmail)
    {
        $this->guardAgainstMaxLength('senderEmail', $senderEmail, $threshold = 90);
        $this->guardAgainstInvalidEmail($senderEmail);
        $this->senderEmail = $senderEmail;

        return $this;
    }

    /**
     * Set the email subject.
     *
     * @param  string $subject
     * @return self
     */
    public function setSubject($subject)
    {
        $this->guardAgainstMaxLength('subject', $subject, $threshold = 256);
        $this->subject = $subject;

        return $this;
    }

    /**
     * Set the email message.
     *
     * @param  string $emailMessage
     * @return self
     */
    public function setEmailMessage($emailMessage)
    {
        $this->guardAgainstMaxLength('emailMessage', $emailMessage, $threshold = 50);
        $this->emailMessage = $emailMessage;

        return $this;
    }

    /**
     * Set the email closing name.
     *
     * @param  string $emailClosingName
     * @return self
     */
    public function setEmailClosingName($emailClosingName)
    {
        $this->guardAgainstMaxLength('emailClosingName', $emailClosingName, $threshold = 200);
        $this->emailClosingName = $emailClosingName;

        return $this;
    }

    /**
     * Set the unsubscribe data.
     *
     * @param array|UnsubscribeData $unsubscribeData
     * @return self
     */
    public function setUnsubscribeData($unsubscribeData)
    {
        if (is_array($unsubscribeData) && ! empty($unsubscribeData)) {
            $unsubscribeData = app(UnsubscribeData::class, $unsubscribeData);
        }

        if (! $unsubscribeData instanceof UnsubscribeData) {
            throw new \InvalidArgumentException("Invalid unsubscribeData, you must pass an instance of UnsubscribeData or an array that can be converted.");
        }

        $this->unsubscribeData = $unsubscribeData;

        return $this;
    }
}
