<?php

namespace Ignite\Vendor\Hawk\Response;

use Ignite\Vendor\Hawk\ResponseException;
use Ignite\Vendor\Hawk\Status;
use Illuminate\Contracts\Support\Arrayable;
use Psr\Http\Message\ResponseInterface;

/**
 * Class TransactionResponse
 *
 * @method string getStartTime()
 * @method string getCompletionTime()
 * @method string getVersionId()
 * @method string getStatus()
 * @method bool getIsReversible()
 * @method bool getWasReversed()
 * @method array getContext()
 * @method array getRequest()
 * @method array getResponse()
 * @method int getStepsCompleted()
 * @method bool getSteps()
 */
abstract class TransactionResponse extends Response implements Arrayable
{
    /**
     * @var array
     */
    protected $data;

    /**
     * Create an instance from a Http Response.
     *
     * @param ResponseInterface $response
     * @return SubmitEGiftIndividualTransactionResponse|static
     * @throws ResponseException
     */
    public static function fromHttpResponse(ResponseInterface $response)
    {
        $data = static::parseJson($response);

        return app(static::class, compact('data'))->setHttpResponse($response);
    }

    /**
     * TransactionResponse constructor.
     *
     * @param array $data
     */
    public function __construct(array $data)
    {
        $this->data = $data;
    }

    /**
     * Call to get a property dynamically given the property actually exists.
     *
     * @param  string $method
     * @param  array  $arguments
     * @return mixed
     */
    public function __call($method, $arguments)
    {
        $property = $this->camelCase(substr($method, 3));

        if (isset($this->data[$property])) {
            return $this->data[$property];
        }

        return parent::__call($method, $arguments);
    }

    /**
     * Determine whether the transaction is completed.
     *
     * @return bool
     */
    public function isCompleted()
    {
        return $this->data['status'] === Status::COMPLETED;
    }

    /**
     * Determine whether the transaction is queued.
     *
     * @return bool
     */
    public function isQueued()
    {
        return $this->data['status'] === Status::QUEUED;
    }

    /**
     * Determine whether the transaction is processing.
     *
     * @return bool
     */
    public function isProcessing()
    {
        return $this->data['status'] === Status::PROCESSING;
    }

    /**
     * Determine whether the transaction is failed.
     *
     * @return bool
     */
    public function isFailed()
    {
        return $this->data['status'] === Status::FAILED;
    }

    /**
     * The response as an array.
     *
     * @return array
     */
    public function responseToArray()
    {
        if (empty($this->data['response'])) {
            return [];
        }

        return $this->data['response'];
    }

    /**
     * Json encode the response array.
     *
     * @return false|string
     */
    public function responseToJson()
    {
        return json_encode($this->responseToArray(), JSON_PRETTY_PRINT);
    }
}
