<?php

namespace Ignite\Vendor\Hawk;

use Illuminate\Contracts\Support\Arrayable;

class FundingOrderLine extends Model implements Arrayable
{
    /** @var PersonalizedRecipient */
    protected $recipient;

    /** @var float */
    protected $amount;

    /** @var string */
    protected $shippingMethod;

    /** @var string */
    protected $cardCarrierMessage;

    /** @var string */
    protected $description;

    /** @var string */
    protected $fourthLineEmbossText;

    /** @var array */
    protected $additionalInfo;

    /**
     * Create an instance from an associative array.
     *
     * @param  array $data
     * @return FundingOrderLine
     */
    public static function fromAssociativeArray($data)
    {
        return app(static::class, $data);
    }

    /**
     * FundingOrderLine constructor.
     *
     * @param array|PersonalizedRecipient $recipient
     * @param float                       $amount
     * @param null|string                 $description
     * @param array                       $additionalInfo
     */
    public function __construct(
        $recipient,
        $amount,
        $description = null,
        $additionalInfo = []
    ) {
        // https://developer.blackhawknetwork.com/hawkmarketplace/reference/dosubmitfunding
        $this->setRecipient($recipient);
        $this->setAmount($amount);
        $this->setDescription($description);
        $this->additionalInfo = $additionalInfo;
    }

    /**
     * Set the recipient.
     *
     * @param  array|PersonalizedRecipient $recipient
     * @return $this
     */
    public function setRecipient($recipient)
    {
        if (is_array($recipient) && ! empty($recipient)) {
            $recipient = app(Recipient::class, $recipient);
        }

        if (! $recipient instanceof Recipient) {
            throw new \InvalidArgumentException("Invalid recipient, you must pass an instance of PersonalizedRecipient or an array that can be converted.");
        }

        $this->recipient = $recipient;

        return $this;
    }

    /**
     * Set the amount.
     *
     * @param  mixed $amount
     * @return $this
     */
    private function setAmount($amount)
    {
        if (! is_numeric($amount)) {
            throw new \InvalidArgumentException("Invalid amount, must be numeric: $amount");
        }

        $this->amount = (float) $amount;

        return $this;
    }

    /**
     * Set the fourth line emboss text.
     *
     * @param  string $description
     * @return $this
     */
    private function setDescription($description)
    {
        if (! is_null($description)) {
            $this->guardAgainstMaxLength('description', $description, 40);
        }

        $this->description = $description;

        return $this;
    }
}
