<?php

namespace Ignite\Vendor\Hawk;

use InvalidArgumentException;

class UnsubscribeData extends Model
{
    const METHOD_EMAIL = 'EMAIL';
    const METHOD_WEBSITE = 'WEBSITE';
    const COUNTRY_US = 'UNITED STATES';
    const COUNTRY_CA = 'CANADA';

    /**
     * @var string
     */
    protected $companyName;

    /**
     * @var string
     */
    protected $addressLine1;

    /**
     * @var string
     */
    protected $addressCity;

    /**
     * @var string
     */
    protected $addressState;

    /**
     * @var string
     */
    protected $addressZip;

    /**
     * @var string
     */
    protected $addressCountry;

    /**
     * @var string
     */
    protected $methodType;

    /**
     * @var string
     */
    protected $value;

    /**
     * VirtualRecipient constructor.
     *
     * @param string $companyName
     * @param string $addressLine1
     * @param string $addressCity
     * @param string $addressState
     * @param string $addressZip
     * @param string $addressCountry
     * @param string $methodType
     * @param string $value
     */
    public function __construct(
        $companyName,
        $addressLine1,
        $addressCity,
        $addressState,
        $addressZip,
        $addressCountry,
        $methodType,
        $value
    ) {
        $this->setCompanyName($companyName);
        $this->setAddressLine1($addressLine1);
        $this->setAddressCity($addressCity);
        $this->setAddressCountry($addressCountry);
        $this->setAddressState($addressState);
        $this->setAddressZip($addressZip);
        $this->setMethodType($methodType);
        $this->setValue($value);
    }

    /**
     * Create a new instance from an associative array.
     *
     * @param  array $data
     * @return static
     */
    public static function fromAssociativeArray(array $data)
    {
        return app(static::class, $data);
    }

    /**
     * Set the company name.
     *
     * @param  string $companyName
     * @return self
     */
    public function setCompanyName($companyName)
    {
        $this->guardAgainstMaxLength('companyName', $companyName, $threshold = 60);
        $this->companyName = $companyName;

        return $this;
    }

    /**
     * Set the address line 1.
     *
     * @param  string $addressLine1
     * @return self
     */
    public function setAddressLine1($addressLine1)
    {
        $this->guardAgainstMaxLength('addressLine1', $addressLine1, $threshold = 200);
        $this->addressLine1 = $addressLine1;

        return $this;
    }

    /**
     * Set the address city.
     *
     * @param  string $addressCity
     * @return self
     */
    public function setAddressCity($addressCity)
    {
        $this->guardAgainstMaxLength('addressCity', $addressCity, $threshold = 100);
        $this->addressCity = $addressCity;

        return $this;
    }

    /**
     * Set the email message.
     *
     * @param  string $addressState
     * @return self
     */
    public function setAddressState($addressState)
    {
        $threshold = $this->addressCountry === static::COUNTRY_US ? 2 : 36;
        $this->guardAgainstMaxLength('addressState', $addressState, $threshold);
        $this->addressState = $addressState;

        return $this;
    }

    /**
     * Set the email closing name.
     *
     * @param  string $addressZip
     * @return self
     */
    public function setAddressZip($addressZip)
    {
        $threshold = $this->addressCountry === static::COUNTRY_US ? 5 : 11;
        $this->guardAgainstMaxLength('addressZip', $addressZip, $threshold);
        $this->addressZip = $addressZip;

        return $this;
    }

    /**
     * Set the unsubscribe data.
     *
     * @param  string $addressCountry
     * @return self
     */
    public function setAddressCountry($addressCountry)
    {
        $supportedCountries = [
            'US' => static::COUNTRY_US,
            'CA' => static::COUNTRY_CA,
        ];

        if (! array_key_exists($addressCountry, $supportedCountries)) {
            $countries = implode(', ', array_keys($supportedCountries));
            throw new InvalidArgumentException(
                "Invalid address country. $addressCountry. Supported: $countries"
            );
        }

        $this->addressCountry = $supportedCountries[$addressCountry];

        return $this;
    }

    /**
     * Set the method type.
     *
     * @param  string $methodType
     * @return self
     */
    private function setMethodType($methodType)
    {
        $types = [
            static::METHOD_EMAIL,
            static::METHOD_WEBSITE,
        ];

        if (! in_array($methodType, $types)) {
            $types = implode(', ', $types);
            throw new InvalidArgumentException(
                "Invalid method type. $methodType. Types: $types"
            );
        }

        $this->methodType = $methodType;

        return $this;
    }

    /**
     * Set the value.
     *
     * @param  string $value
     * @return self
     */
    private function setValue($value)
    {
        $this->guardAgainstMaxLength('value', $value, 255);
        $this->value = $value;

        return $this;
    }
}
