<?php

namespace Ignite\Vendor\Hawk;

use Illuminate\Contracts\Support\Arrayable;

class OpenSubmitLoopAnonymousBulkOrderLine extends Model implements Arrayable
{
    /**
     * @var int
     */
    protected $amount;

    /**
     * @var int
     */
    protected $quantity;

    /**
     * @var OpenLoopAnonymousDeliveryAddress
     */
    protected $deliveryAddress;

    /**
     * @var string
     */
    protected $contentProvider;

    /**
     * @var string
     */
    protected $description;

    /**
     * @var string
     */
    protected $clientRefId;

    /**
     * @var string
     */
    protected $shippingMethod;

    /**
     * @var string
     */
    protected $cardCarrierMessage;

    /**
     * @var string
     */
    protected $fourthLineEmbossText;

    /**
     * @var array
     */
    protected $additionalInfo;

    /**
     * Create an instance from an associative array, assuming that array has the required keys.
     *
     * @param array $data
     * @return static
     */
    public static function fromAssociativeArray(array $data)
    {
        return app(static::class, $data);
    }

    /**
     * OpenSubmitLoopAnonymousBulkOrderLine constructor.
     *
     * @param int $amount
     * @param array|DeliveryAddress $deliveryAddress
     * @param int $quantity
     * @param string|null $description
     * @param string|null $clientRefId
     * @param string|null $fourthLineEmbossText
     * @param string|null $cardCarrierMessage
     * @param string $shippingMethod
     * @param array $additionalInfo
     */
    public function __construct(
        int $amount,
        $deliveryAddress,
        int $quantity,
        $description = '',
        $clientRefId = '',
        $fourthLineEmbossText = '',
        $cardCarrierMessage = '',
        string $shippingMethod = ShippingMethod::UPS_GROUND,
        array $additionalInfo = []
    ) {
        // https://developer.blackhawknetwork.com/hawkmarketplace/reference/dosubmitopenloopanonymous
        $this->setAmount($amount);
        $this->setDeliveryAddress($deliveryAddress);
        $this->setQuantity($quantity);
        $this->setDescription($description);
        $this->setClientRefId($clientRefId);
        $this->setFourthLineEmbossText($fourthLineEmbossText);
        $this->setCardCarrierMessage($cardCarrierMessage);
        $this->setShippingMethod($shippingMethod);
        $this->setAdditionalInfo($additionalInfo);
    }

    /**
     * @param array $additionalInfo
     */
    protected function setAdditionalInfo(array $additionalInfo)
    {
        $this->additionalInfo = $additionalInfo;
        return $this;
    }

    /**
     * @param int $amount
     */
    protected function setAmount(int $amount)
    {
        $this->amount = $amount;
        return $this;
    }

    /**
     * @param string setCardCarrierMessage
     */
    protected function setCardCarrierMessage(string $cardCarrierMessage = null)
    {
        $this->guardAgainstMaxLength('cardCarrierMessage', $cardCarrierMessage, 80);
        $this->cardCarrierMessage = $cardCarrierMessage;
        return $this;
    }

    /**
     * @param string $clientRefId
     */
    protected function setClientRefId(string $clientRefId)
    {
        $this->clientRefId = $clientRefId;
        return $this;
    }

    /**
     * Set the delivery address.
     *
     * @param array|OpenLoopAnonymousDeliveryAddress $deliveryAddress
     * @return $this
     */
    protected function setDeliveryAddress($deliveryAddress)
    {
        if (is_array($deliveryAddress) && !empty($deliveryAddress)) {
            $deliveryAddress = OpenLoopAnonymousDeliveryAddress::fromAssociativeArray($deliveryAddress);
        }

        if (!$deliveryAddress instanceof OpenLoopAnonymousDeliveryAddress) {
            throw new \InvalidArgumentException(
                "Invalid deliveryAddress, you must pass an instance of"
                . " OpenLoopAnonymousDeliveryAddress or an array that can be converted."
            );
        }

        $this->deliveryAddress = $deliveryAddress;

        return $this;
    }

    /**
     * @param string $description
     */
    protected function setDescription(string $description)
    {
        $this->guardAgainstMaxLength('description', $description, 40);
        $this->$description = $description;
        return $this;
    }

    /**
     * @param string $fourthLineEmbossText
     */
    protected function setFourthLineEmbossText(string $fourthLineEmbossText = null)
    {
        $this->guardAgainstMaxLength('fourthLineEmbossText', $fourthLineEmbossText, 21);
        $this->fourthLineEmbossText = $fourthLineEmbossText;
        return $this;
    }

    /**
     * @param int $quantity
     */
    protected function setQuantity(int $quantity)
    {
        $this->quantity = $quantity;
        return $this;
    }

    /**
     * @param string $shippingMethod
     */
    protected function setShippingMethod(string $shippingMethod)
    {
        $this->shippingMethod = $shippingMethod;
        return $this;
    }
}
