<?php

namespace Ignite\Vendor\Hawk\Response;

use Illuminate\Contracts\Support\Arrayable;
use Ignite\Vendor\Hawk\ResponseException;
use Psr\Http\Message\ResponseInterface;

/**
 * Class BaseOrderResponse
 *
 * @method string getTransactionId()
 * @method string getOrderNumber()
 * @method bool getIsCompleted()
 * @method int getPercentComplete()
 * @method bool getSuccess()
 * @method string hasTransactionId()
 * @method string hasOrderNumber()
 * @method bool hasIsCompleted()
 * @method int hasPercentComplete()
 * @method bool hasSuccess()
 */
abstract class BaseOrderResponse extends Response implements Arrayable
{
    /**
     * @var string
     */
    protected $transactionId;

    /**
     * @var string
     */
    protected $orderNumber;

    /**
     * @var bool
     */
    protected $isCompleted;

    /**
     * @var int
     */
    protected $percentComplete;

    /**
     * @var bool
     */
    protected $success;

    /**
     * Create an instance from a Http Response.
     *
     * @param ResponseInterface $response
     * @return SubmitClosedLoopAnonymousResponse|static
     * @throws ResponseException
     */
    public static function fromHttpResponse(ResponseInterface $response)
    {
        $data = static::parseJson($response);

        if (! isset($data['transactionId'])) {
            throw new ResponseException($response, 'Expected a transaction ID in the response, unable to find the value using key `transactionId`.');
        }

        if (! isset($data['isCompleted'])) {
            throw new ResponseException($response, 'Expected indication of completion in response, unable to find the value using key `isCompleted`.');
        }

        if (! isset($data['percentComplete'])) {
            throw new ResponseException($response, 'Expected an indication of percentage complete in the response, unable to find the value using key `percentComplete`.');
        }

        if (! isset($data['success'])) {
            throw new ResponseException($response, 'Expected indication of success or failure in response, unable to find the value using key `success`.');
        }

        if (! isset($data['orderNumber'])) {
            $data['orderNumber'] = 'Unavailable';
            //throw new ResponseException($response, 'Expected to find an order number in response, unable to find the value using key `orderNumber`.');
        }

        return app(static::class, $data)->setHttpResponse($response);
    }

    /**
     * SubmitClosedLoopAnonymousResponse constructor.
     *
     * @param string $transactionId
     * @param string $orderNumber
     * @param bool   $isCompleted
     * @param int    $percentComplete
     * @param bool   $success
     */
    public function __construct(
        string $transactionId,
        string $orderNumber,
        bool $isCompleted = false,
        int $percentComplete = 0,
        bool $success = false
    ) {
        $this->transactionId = $transactionId;
        $this->orderNumber = $orderNumber;
        $this->isCompleted = $isCompleted;
        $this->percentComplete = $percentComplete;
        $this->success = $success;
    }
}
