<?php

namespace Ignite\Vendor\Hawk;

class RecipientAddress extends Model
{
    /**
     * @var string
     */
    protected $line1;

    /**
     * @var string
     */
    protected $city;

    /**
     * @var string
     */
    protected $region;

    /**
     * @var string
     */
    protected $postalCode;

    /**
     * @var string
     */
    protected $country;

    /**
     * @var string
     */
    protected $line2;

    /**
     * @var string
     */
    protected $line3;

    /**
     * @var string
     */
    protected $postalCodeExt;

    /**
     * Create an instance from an associative array, assuming that array has the required keys.
     *
     * @param  array $data
     * @return $this
     */
    public static function fromAssociativeArray($data)
    {
        return app(static::class, $data);
    }

    /**
     * RecipientAddress constructor.
     *
     * @param string $line1
     * @param string $city
     * @param string $region
     * @param string $postalCode
     * @param string $country
     * @param string $line2
     * @param string $line3
     * @param string $postalCodeExt
     */
    public function __construct(
        $line1,
        $city,
        $region,
        $postalCode,
        $country,
        $line2 = null,
        $line3 = null,
        $postalCodeExt = null
    ) {
        $this->setLine1($line1);
        $this->setCity($city);
        $this->setRegion($region);
        $this->setCountry($country);
        $this->setPostal($postalCode);
        $this->setLine2($line2);
        $this->setLine3($line3);
        $this->setPostalCodeExt($postalCodeExt);
    }

    /**
     * Get the instance as an array.
     *
     * @return array
     */
    public function toArray()
    {
        return array_filter(parent::toArray());
    }

    /**
     * Set the address line 1.
     *
     * @param  string $line1
     * @return $this
     */
    private function setLine1($line1)
    {
        $this->guardAgainstMaxLength('line1', $line1, 35);
        $this->line1 = $line1;

        return $this;
    }

    /**
     * Set the address line 2.
     *
     * @param  string $line2
     * @return $this
     */
    private function setLine2($line2)
    {
        $this->guardAgainstMaxLength('line2', $line2, 35);
        $this->line2 = $line2;

        return $this;
    }

    /**
     * Set the address line 3.
     *
     * @param  string $line3
     * @return $this
     */
    private function setLine3($line3)
    {
        $this->guardAgainstMaxLength('line3', $line3, 35);
        $this->line3 = $line3;

        return $this;
    }

    /**
     * Set the city.
     *
     * @param  string $city
     * @return $this
     */
    private function setCity($city)
    {
        $this->guardAgainstMaxLength('city', $city, 35);
        $this->city = $city;

        return $this;
    }

    /**
     * Set the city.
     *
     * @param  string $region
     * @return $this
     */
    private function setRegion($region)
    {
        // Address region must be a valid 2-character uppercase abbreviation for a U.S. state or a Canadian province
        // or contain an international region less than 36 characters.
        $threshold = $this->country === 'USA' ? 2 : 35;

        $this->guardAgainstMaxLength('region', $region, $threshold);
        $this->region = $region;

        return $this;
    }

    /**
     * Set the postal code.
     *
     * @param  string $postalCode
     * @return $this
     */
    private function setPostal($postalCode)
    {
        $this->guardAgainstMaxLength('postalCode', $postalCode, 10);
        $this->postalCode = $postalCode;

        return $this;
    }

    /**
     * Set the postal code.
     *
     * @param  string $country
     * @return $this
     */
    private function setCountry($country)
    {
        $this->guardAgainstMaxLength('country', $country, 3);
        $supported = ['USA', 'CAN'];
        if (! in_array($country, $supported)) {
            throw new \InvalidArgumentException("Unsupported country: $country");
        }

        $this->country = $country;

        return $this;
    }

    /**
     * Set the postal code extension.
     *
     * @param string $postalCodeExt
     * @return $this
     */
    private function setPostalCodeExt($postalCodeExt)
    {
        $this->guardAgainstMaxLength('postalCodeExt', $postalCodeExt, 4);
        $this->postalCodeExt = $postalCodeExt;

        return $this;
    }
}
