<?php

namespace Ignite\Vendor\Hawk;

use Illuminate\Contracts\Support\Arrayable;

/**
 * Delivery Address Model.
 *
 * @method string getCompany()
 * @method string getFirstName()
 * @method string getLastName()
 * @method string getLine1()
 * @method string getLine2()
 * @method string getLine3()
 * @method string getCity()
 * @method string getRegion()
 * @method string getPostalCode()
 * @method string getCountry()
 * @method string getPostalCodeExt()
 * @method string hasCompany()
 * @method string hasLine1()
 * @method string hasLine2()
 * @method string hasLine3()
 * @method string hasCity()
 * @method string hasRegion()
 * @method string hasPostalCode()
 * @method string hasCountry()
 * @method string hasPostalCodeExt()
 */
class OpenLoopAnonymousDeliveryAddress extends Model implements Arrayable
{
    /**
     * @var string
     */
    protected $company;

    /**
     * @var string
     */
    protected $firstName;

    /**
     * @var string
     */
    protected $lastName;

    /**
     * @var string
     */
    protected $line1;

    /**
     * @var string
     */
    protected $city;

    /**
     * @var string
     */
    protected $region;

    /**
     * @var string
     */
    protected $postalCode;

    /**
     * @var string
     */
    protected $country;

    /**
     * @var string
     */
    protected $line2;

    /**
     * @var string
     */
    protected $line3;

    /**
     * @var string
     */
    protected $postalCodeExt;

    /**
     * Create an instance from an associative array, assuming that array has the required keys.
     *
     * @param array $data
     * @return $this
     */
    public static function fromAssociativeArray($data)
    {
        return app(static::class, $data);
    }

    /**
     * DeliveryAddress constructor.
     *
     * @param string $company
     * @param string $firstName
     * @param string $lastName
     * @param string $line1
     * @param string $city
     * @param string $region
     * @param string $postalCode
     * @param string $country
     * @param string|null $line2
     * @param string|null $line3
     * @param string|null $postalCodeExt
     */
    public function __construct(
        string $company,
        string $firstName,
        string $lastName,
        string $line1,
        string $city,
        string $region,
        string $postalCode,
        string $country,
        $line2 = '',
        $line3 = '',
        $postalCodeExt = null
    ) {
        // https://developer.blackhawknetwork.com/hawkmarketplace/reference/dosubmitopenloopanonymous
        $this->setCompany($company);
        $this->setFirstName($firstName);
        $this->setLastName($lastName);
        $this->setLine1($line1);
        $this->setCity($city);
        $this->setRegion($region);
        $this->setPostalCode($postalCode);
        $this->setCountry($country);
        $this->setLine2($line2);
        $this->setLine3($line3);
        $this->setPostalCodeExt($postalCodeExt);
    }

    /**
     * @param string $city
     */
    protected function setCity(string $city)
    {
        $this->city = $this->transformExceedingMaxLength('city', $city, 30);
        return $this;
    }

    /**
     * @param string $company
     */
    protected function setCompany(string $company)
    {
        $this->company = $this->transformExceedingMaxLength('company', $company, 35);
        return $this;
    }

    /**
     * @param string $country
     */
    protected function setCountry(string $country)
    {
        $this->guardAgainstMaxLength('country', $country, 3);
        $this->country = $country;
        return $this;
    }

    /**
     * @param string $firstName
     */
    protected function setFirstName(string $firstName)
    {
        $this->firstName = $this->transformExceedingMaxLength('firstName', $firstName, 45);
        return $this;
    }

    /**
     * @param string $lastName
     */
    protected function setLastName(string $lastName)
    {
        $this->lastName = $this->transformExceedingMaxLength('lastName', $lastName, 45);
        return $this;
    }

    /**
     * @param string $line1
     */
    protected function setLine1(string $line1)
    {
        $this->line1 = $this->guardAgainstMaxLength('line1', $line1, 35);
        return $this;
    }

    /**
     * @param string $line2
     */
    protected function setLine2(string $line2 = null)
    {
        $this->line2 = $this->guardAgainstMaxLength('line2', $line2, 35);
        return $this;
    }

    /**
     * @param string $line3
     */
    protected function setLine3(string $line3 = null)
    {
        $this->line3 = $this->guardAgainstMaxLength('line3', $line3, 35);
        return $this;
    }

    /**
     * @param string $postalCode
     */
    protected function setPostalCode(string $postalCode)
    {
        $this->guardAgainstMaxLength('postalCode', $postalCode, 10);
        $this->postalCode = $postalCode;
        return $this;
    }

    /**
     * @param string $postalCodeExt
     */
    protected function setPostalCodeExt(string $postalCodeExt = null)
    {
        $this->guardAgainstMaxLength('postalCodeExt', $postalCodeExt, 4);
        $this->postalCodeExt = $postalCodeExt;
        return $this;
    }

    /**
     * @param string $region
     */
    protected function setRegion(string $region)
    {
        $this->guardAgainstMaxLength('region', $region, 2);
        $this->region = $region;
        return $this;
    }
}
