<?php

namespace Ignite\Vendor\Hawk;

use GuzzleHttp\Client as Guzzle;
use GuzzleHttp\Exception\GuzzleException;
use Ignite\Vendor\Hawk\Request\Request;
use Ignite\Vendor\Hawk\Request\SubmitClosedLoopBulkRequest;
use Ignite\Vendor\Hawk\Request\SubmitEGiftIndividualRequest;
use Ignite\Vendor\Hawk\Request\SubmitClosedLoopAnonymousRequest;
use Ignite\Vendor\Hawk\Request\SubmitClosedLoopIndividualRequest;
use Ignite\Vendor\Hawk\Request\SubmitFundingRequest;
use Ignite\Vendor\Hawk\Request\SubmitOpenLoopAnonymousRequest;
use Ignite\Vendor\Hawk\Request\SubmitOpenLoopPersonalizedIndividualRequest;
use Ignite\Vendor\Hawk\Request\SubmitVirtualIndividualRequest;
use Ignite\Vendor\Hawk\Response\ClientProgramResponse;
use Ignite\Vendor\Hawk\Response\ErrorResponse;
use Ignite\Vendor\Hawk\Response\OrderInfoResponse;
use Ignite\Vendor\Hawk\Response\SubmitClosedLoopIndividualTransactionResponse;
use Ignite\Vendor\Hawk\Response\SubmitFundingResponse;
use Ignite\Vendor\Hawk\Response\SubmitClosedLoopBulkResponse;
use Ignite\Vendor\Hawk\Response\SubmitEGiftIndividualResponse;
use Ignite\Vendor\Hawk\Response\SubmitClosedLoopAnonymousResponse;
use Ignite\Vendor\Hawk\Response\SubmitClosedLoopIndividualResponse;
use Ignite\Vendor\Hawk\Response\SubmitEGiftIndividualTransactionResponse;
use Ignite\Vendor\Hawk\Response\SubmitFundingTransactionResponse;
use Ignite\Vendor\Hawk\Response\SubmitOpenLoopAnonymousResponse;
use Ignite\Vendor\Hawk\Response\SubmitOpenLoopAnonymousTransactionResponse;
use Ignite\Vendor\Hawk\Response\SubmitOpenLoopPersonalizedIndividualResponse;
use Ignite\Vendor\Hawk\Response\SubmitOpenLoopPersonalizedIndividualTransactionResponse;
use Ignite\Vendor\Hawk\Response\SubmitVirtualIndividualResponse;
use Ignite\Vendor\Hawk\Response\SubmitVirtualIndividualTransactionResponse;
use Psr\Http\Message\ResponseInterface;

class Client
{
    /**
     * @var string
     */
    protected $merchantId;

    /**
     * @var Ssl
     */
    protected $ssl;

    /**
     * @var Client
     */
    protected $client;

    /**
     * The production endpoint.
     *
     * @var string
     */
    protected $liveEndpoint = 'https://api.blackhawknetwork.com';

    /**
     * The testing endpoint.
     *
     * @var string
     */
    protected $testEndpoint = 'https://apipp.blackhawknetwork.com';

    /**
     * The rewards endpoint paths.
     *
     * @var array
     */
    protected $paths = [
        'orders' => '/rewardsOrderProcessing/v1/',
        'catalog' => '/rewardsCatalogProcessing/v1/',
    ];

    /**
     * Flag to indicate if we are in test mode.
     *
     * @var bool
     */
    protected $testMode = false;

    /**
     * The current count of attempts.
     *
     * @var int
     */
    protected static $attempts = 0;

    /**
     * HawkApi constructor.
     *
     * @param string      $merchantId
     * @param Ssl         $ssl
     * @param Guzzle|null $client
     * @param bool        $testMode
     */
    public function __construct($merchantId, Ssl $ssl, Guzzle $client = null, $testMode = false)
    {
        $this->merchantId = $merchantId;
        $this->ssl = $ssl;
        $this->testMode = (bool) $testMode;
        $this->client = is_null($client) ? $this->getDefaultClient() : $client;
    }

    /**
     * The path for the endpoint.
     *
     * @param string $type
     * @param string $endpoint
     *
     * @return string
     */
    protected function path($endpoint, $type = 'orders')
    {
        if (! array_key_exists($type, $this->paths)) {
            $type = 'orders';
        }

        return $this->paths[$type] . $endpoint;
    }

    /**
     * Get the client program information.
     *
     * @tag beta
     * @param string $clientProgramId
     * @param string|null $requestId
     *
     * @return ErrorResponse|ClientProgramResponse
     * @throws GuzzleException
     * @throws ResponseException
     */
    public function clientProgram($clientProgramId, $requestId = null)
    {
        $query = array_filter([
            'clientProgramId' => $clientProgramId,
        ]);

        $method = $this->path('clientProgram/byKey', 'catalog');
        $response = $this->buildGetRequest($method, $query, $requestId);

        if ($this->isErrorCode($response)) {
            return ErrorResponse::fromHttpResponse($response);
        }

        return ClientProgramResponse::fromHttpResponse($response);
    }

    /**
     * Submit an giftcard individual order.
     *
     * @param  SubmitEGiftIndividualRequest $request
     * @param  string|null                  $requestId
     * @param  int                          $millisecondsToWait
     *
     * @return SubmitEgiftIndividualResponse|ErrorResponse
     * @throws ResponseException
     * @throws GuzzleException
     */
    public function submitEgiftIndividual(SubmitEGiftIndividualRequest $request, $requestId = null, $millisecondsToWait = 30000)
    {
        $method = $this->path('submitEgiftIndividual');
        $response = $this->buildPostRequest($method, $request, $requestId, $millisecondsToWait);

        if ($this->isErrorCode($response)) {
            return ErrorResponse::fromHttpResponse($response);
        }

        return SubmitEGiftIndividualResponse::fromHttpResponse($response);
    }

    /**
     * Submit an giftcard individual order.
     *
     * @param  string $transactionId
     * @param  string|null $requestId
     * @param  int    $millisecondsToWait
     *
     * @return SubmitEGiftIndividualTransactionResponse|ErrorResponse
     * @throws ResponseException
     * @throws GuzzleException
     */
    public function submitEgiftIndividualTransaction($transactionId, $requestId = null, $millisecondsToWait = 30000)
    {
        $method = $this->path("submitEgiftIndividual/{$transactionId}");
        $response = $this->buildGetRequest($method, [], $requestId, $millisecondsToWait);

        if ($this->isErrorCode($response)) {
            return ErrorResponse::fromHttpResponse($response);
        }

        return SubmitEGiftIndividualTransactionResponse::fromHttpResponse($response);
    }

    /**
     * Submit a closed loop individual order. Giftcard.
     *
     * @param  SubmitClosedLoopIndividualRequest $request
     * @param  string|null $requestId
     * @param  int $millisecondsToWait
     *
     * @return ErrorResponse|SubmitClosedLoopIndividualResponse|static
     * @throws ResponseException
     * @throws GuzzleException
     */
    public function submitClosedLoopIndividual(SubmitClosedLoopIndividualRequest $request, $requestId = null, $millisecondsToWait = 30000)
    {
        $method = $this->path('submitClosedLoopIndividual');
        $response = $this->buildPostRequest($method, $request, $requestId, $millisecondsToWait);

        if ($this->isErrorCode($response)) {
            return ErrorResponse::fromHttpResponse($response);
        }

        return SubmitClosedLoopIndividualResponse::fromHttpResponse($response);
    }

    /**
     * Submit an giftcard individual order.
     *
     * @param  string $transactionId
     * @param  string|null $requestId
     * @param  int    $millisecondsToWait
     *
     * @return SubmitClosedLoopIndividualTransactionResponse|ErrorResponse
     * @throws ResponseException
     * @throws GuzzleException
     */
    public function submitClosedLoopIndividualTransaction($transactionId, $requestId = null, $millisecondsToWait = 30000)
    {
        $method = $this->path("submitClosedLoopIndividual/{$transactionId}");
        $response = $this->buildGetRequest($method, [], $requestId, $millisecondsToWait);

        if ($this->isErrorCode($response)) {
            return ErrorResponse::fromHttpResponse($response);
        }

        return SubmitClosedLoopIndividualTransactionResponse::fromHttpResponse($response);
    }

    /**
     * Submit a closed loop bulk order. Giftcard, multiple quantities.
     *
     * @param  SubmitClosedLoopBulkRequest $request
     * @param  string|null $requestId
     * @param  int $millisecondsToWait
     *
     * @return ErrorResponse|SubmitClosedLoopBulkResponse|static
     * @throws ResponseException
     * @throws GuzzleException
     */
    public function submitClosedLoopBulk(SubmitClosedLoopBulkRequest $request, $requestId = null, $millisecondsToWait = 30000)
    {
        $method = $this->path('submitClosedLoopBulk');
        $response = $this->buildPostRequest($method, $request, $requestId, $millisecondsToWait);

        if ($this->isErrorCode($response)) {
            return ErrorResponse::fromHttpResponse($response);
        }

        return SubmitClosedLoopBulkResponse::fromHttpResponse($response);
    }

    /**
     *
     * @param  string $transactionId
     * @param  string|null $requestId
     * @param  int    $millisecondsToWait
     *
     * @return SubmitOpenLoopAnonymousResponse|ErrorResponse
     * @throws ResponseException
     * @throws GuzzleException
     */
    public function submitOpenLoopAnonymous(SubmitOpenLoopAnonymousRequest $request, $requestId = null, $millisecondsToWait = 30000)
    {
        $method = $this->path("submitOpenLoopAnonymous");
        $response = $this->buildPostRequest($method, $request, $requestId, $millisecondsToWait);

        if ($this->isErrorCode($response)) {
            return ErrorResponse::fromHttpResponse($response);
        }

        return SubmitOpenLoopAnonymousResponse::fromHttpResponse($response);
    }

    /**
     *
     * @param string $transactionId
     * @param string|null $requestId
     * @param int $millisecondsToWait
     *
     * @return ErrorResponse|SubmitOpenLoopAnonymousTransactionResponse
     * @throws ResponseException
     * @throws GuzzleException
     */
    public function submitOpenLoopAnonymousTransactionResponse(string $transactionId, string $requestId = null, int $millisecondsToWait = 30000)
    {
        $method = $this->path("submitOpenLoopAnonymous/{$transactionId}");
        $response = $this->buildGetRequest($method, [], $requestId, $millisecondsToWait);

        if ($this->isErrorCode($response)) {
            return ErrorResponse::fromHttpResponse($response);
        }

        return SubmitOpenLoopAnonymousTransactionResponse::fromHttpResponse($response);
    }

    /**
     * Submit a closed loop order anonymously.
     *
     * @param  SubmitClosedLoopAnonymousRequest $request
     * @param  string|null $requestId
     * @param  int $millisecondsToWait
     *
     * @return ErrorResponse|SubmitClosedLoopAnonymousResponse
     * @throws ResponseException
     * @throws GuzzleException
     */
    public function submitClosedLoopAnonymous(SubmitClosedLoopAnonymousRequest $request, $requestId = null, $millisecondsToWait = 30000)
    {
        $method = $this->path('submitClosedLoopAnonymous');
        $response = $this->buildPostRequest($method, $request, $requestId, $millisecondsToWait);

        if ($this->isErrorCode($response)) {
            return ErrorResponse::fromHttpResponse($response);
        }

        return SubmitClosedLoopAnonymousResponse::fromHttpResponse($response);
    }

    /**
     * Submit a virtual individual order.
     *
     * @param SubmitVirtualIndividualRequest $request
     * @param null $requestId
     * @param int $millisecondsToWait
     *
     * @return ErrorResponse|SubmitVirtualIndividualResponse
     * @throws GuzzleException
     * @throws ResponseException
     */
    public function submitVirtualIndividual(SubmitVirtualIndividualRequest $request, $requestId = null, $millisecondsToWait = 30000)
    {
        $method = $this->path('submitVirtualIndividual');
        $response = $this->buildPostRequest($method, $request, $requestId, $millisecondsToWait);

        if ($this->isErrorCode($response)) {
            return ErrorResponse::fromHttpResponse($response);
        }

        return SubmitVirtualIndividualResponse::fromHttpResponse($response);
    }

    /**
     * Submit a virtual individual order.
     *
     * @param  string $transactionId
     * @param  string|null $requestId
     * @param  int    $millisecondsToWait
     *
     * @return ErrorResponse|SubmitVirtualIndividualTransactionResponse
     * @throws GuzzleException
     * @throws ResponseException
     */
    public function submitVirtualIndividualTransaction($transactionId, $requestId = null, $millisecondsToWait = 30000)
    {
        $method = $this->path("submitClosedLoopIndividual/{$transactionId}");
        $response = $this->buildGetRequest($method, [], $requestId, $millisecondsToWait);

        if ($this->isErrorCode($response)) {
            return ErrorResponse::fromHttpResponse($response);
        }

        return SubmitVirtualIndividualTransactionResponse::fromHttpResponse($response);
    }

    /**
     * Submit an open loop personalized individual (reloadable) orders.
     *
     * @param SubmitOpenLoopPersonalizedIndividualRequest $request
     * @param string|null $requestId
     * @param int $millisecondsToWait
     *
     * @return SubmitOpenLoopPersonalizedIndividualResponse|ErrorResponse
     * @throws ResponseException
     * @throws GuzzleException
     */
    public function submitOpenLoopPersonalizedIndividual(SubmitOpenLoopPersonalizedIndividualRequest $request, $requestId = null, $millisecondsToWait = 30000)
    {
        $method = $this->path('submitOpenLoopPersonalizedIndividual');
        $response = $this->buildPostRequest($method, $request, $requestId, $millisecondsToWait);

        if ($this->isErrorCode($response)) {
            return ErrorResponse::fromHttpResponse($response);
        }

        return SubmitOpenLoopPersonalizedIndividualResponse::fromHttpResponse($response);
    }

    /**
     * Submit an giftcard individual order.
     *
     * @param  string $transactionId
     * @param  string|null $requestId
     * @param  int    $millisecondsToWait
     *
     * @return SubmitOpenLoopPersonalizedIndividualTransactionResponse|ErrorResponse
     * @throws ResponseException
     * @throws GuzzleException
     */
    public function submitOpenLoopPersonalizedIndividualTransaction($transactionId, $requestId = null, $millisecondsToWait = 30000)
    {
        $method = $this->path("submitOpenLoopPersonalizedIndividual/{$transactionId}");
        $response = $this->buildGetRequest($method, [], $requestId, $millisecondsToWait);

        if ($this->isErrorCode($response)) {
            return ErrorResponse::fromHttpResponse($response);
        }

        return SubmitOpenLoopPersonalizedIndividualTransactionResponse::fromHttpResponse($response);
    }

    /**
     * Submit an open loop personalized individual (reloadable) orders.
     *
     * @param SubmitFundingRequest $request
     * @param string|null $requestId
     * @param int $millisecondsToWait
     *
     * @return SubmitFundingResponse|ErrorResponse
     * @throws ResponseException
     * @throws GuzzleException
     */
    public function submitFunding(SubmitFundingRequest $request, $requestId = null, $millisecondsToWait = 20000)
    {
        $method = $this->path('submitFunding');
        $response = $this->buildPostRequest($method, $request, $requestId, $millisecondsToWait);

        if ($this->isErrorCode($response)) {
            return ErrorResponse::fromHttpResponse($response);
        }

        return SubmitFundingResponse::fromHttpResponse($response);
    }

    /**
     * Submit an reloadable individual order.
     *
     * @param  string $transactionId
     * @param  string|null $requestId
     * @param  int    $millisecondsToWait
     *
     * @return SubmitFundingTransactionResponse|ErrorResponse
     * @throws ResponseException
     * @throws GuzzleException
     */
    public function submitFundingTransaction($transactionId, $requestId = null, $millisecondsToWait = 30000)
    {
        $method = $this->path("submitFunding/{$transactionId}");
        $response = $this->buildGetRequest($method, [], $requestId, $millisecondsToWait);

        if ($this->isErrorCode($response)) {
            return ErrorResponse::fromHttpResponse($response);
        }

        return SubmitFundingTransactionResponse::fromHttpResponse($response);
    }

    /**
     * Submit a request to retrieve order information.
     *
     * @param  string|null $orderNumber
     * @param  string|null $clientProgramNumber
     * @param  string|null $orderRequestId
     * @param  string|null $requestId
     * @param  int         $millisecondsToWait
     * @return OrderInfoResponse|ErrorResponse
     *
     * @throws ResponseException
     * @throws GuzzleException
     */
    public function orderInfo($orderNumber = null, $clientProgramNumber = null, $orderRequestId = null, $requestId = null, $millisecondsToWait = 20000)
    {
        $query = array_filter([
            'orderNumber' => $orderNumber,
            'clientProgramNumber' => $clientProgramNumber,
            'requestId' => $orderRequestId
        ]);

        $method = $this->path('orderInfo/byKeys');
        $response = $this->buildGetRequest($method, $query, $requestId, $millisecondsToWait);

        if ($this->isErrorCode($response)) {
            return ErrorResponse::fromHttpResponse($response);
        }

        return OrderInfoResponse::fromHttpResponse($response);
    }

    /**
     * Build a POST request.
     *
     * @param  string $method
     * @param  Request $request
     * @param  string $requestId
     * @param  int $millisecondsToWait
     *
     * @return ResponseInterface
     * @throws GuzzleException
     */
    protected function buildPostRequest($method, Request $request, $requestId, $millisecondsToWait = 30000)
    {
        static::$attempts++;

        return $this->request('post', $method, [
            'cert' => $this->ssl->getCertPath(),
            'ssl_key' => $this->ssl->getKeyPath(),
            'json' => $request->toArray(),
            'headers' => $this->getDefaultHeaders([
                'RequestId' => $requestId,
                'millisecondsToWait' => $millisecondsToWait,
                'previousAttempts' => static::$attempts,
            ])
        ]);
    }

    /**
     * Build a GET request.
     *
     * @param  string $method
     * @param  array  $query
     * @param  string $requestId
     * @param  int    $millisecondsToWait
     *
     * @return ResponseInterface
     * @throws GuzzleException
     */
    protected function buildGetRequest($method, $query, $requestId, $millisecondsToWait = 30000)
    {
        static::$attempts++;

        return $this->request('GET', $method, [
            'cert' => $this->ssl->getCertPath(),
            'ssl_key' => $this->ssl->getKeyPath(),
            'query' => $query,
            'headers' => $this->getDefaultHeaders([
                'RequestId' => $requestId,
                'millisecondsToWait' => $millisecondsToWait,
                'previousAttempts' => static::$attempts,
            ])
        ]);
    }

    /**
     * Make a request to the API.
     *
     * @param  string $method
     * @param  string $uri
     * @param  array $options
     *
     * @return ResponseInterface
     * @throws GuzzleException
     */
    protected function request($method, $uri, array $options)
    {
        return $this->client->request($method, $uri, $options);
    }

    /**
     * The default headers and possibly merge in additional headers.
     *
     * @param  array $additional
     * @return array
     */
    protected function getDefaultHeaders($additional = [])
    {
        return array_merge([
            'MerchantId' => $this->merchantId, // Identifier issued by BHN [string] {required}
        ], $additional);
    }

    /**
     * The default configured HTTP client.
     *
     * @param array $options
     *
     * @return Guzzle
     */
    protected function getDefaultClient($options = [])
    {
        return new Guzzle(array_merge([
            'base_uri' => $this->testMode === false ? $this->liveEndpoint : $this->testEndpoint,
            'debug' => false,
            'verify' => false,
            'exceptions' => false,
            'curl' => [
                CURLOPT_SSL_VERIFYPEER => false,
                CURLOPT_SSLVERSION => CURL_SSLVERSION_TLSv1_2,
                CURLOPT_SSLCERT => $this->ssl->getCertPath(),
                CURLOPT_SSLKEY => $this->ssl->getKeyPath(),
            ]
        ], $options));
    }

    /**
     * Determine if the response status code is an error code.
     *
     * @param ResponseInterface $response
     *
     * @return bool
     */
    protected function isErrorCode(ResponseInterface $response)
    {
        return in_array($response->getStatusCode(), [400, 404, 500, 502, 503]);
    }
}
