<?php

namespace Ignite\Vendor\Hawk;

use GuzzleHttp\Client as Guzzle;
use Ignite\Vendor\Hawk\Request\Request;
use Ignite\Vendor\Hawk\Request\SubmitClosedLoopBulkRequest;
use Ignite\Vendor\Hawk\Request\SubmitEGiftIndividualRequest;
use Ignite\Vendor\Hawk\Request\SubmitClosedLoopAnonymousRequest;
use Ignite\Vendor\Hawk\Request\SubmitClosedLoopIndividualRequest;
use Ignite\Vendor\Hawk\Request\SubmitFundingRequest;
use Ignite\Vendor\Hawk\Request\SubmitOpenLoopPersonalizedIndividualRequest;
use Ignite\Vendor\Hawk\Response\ErrorResponse;
use Ignite\Vendor\Hawk\Response\OrderInfoResponse;
use Ignite\Vendor\Hawk\Response\SubmitClosedLoopBulkResponse;
use Ignite\Vendor\Hawk\Response\SubmitClosedLoopIndividualResponse;
use Ignite\Vendor\Hawk\Response\SubmitEGiftIndividualResponse;
use Ignite\Vendor\Hawk\Response\SubmitClosedLoopAnonymousResponse;
use Ignite\Vendor\Hawk\Response\SubmitFundingResponse;
use Ignite\Vendor\Hawk\Response\SubmitOpenLoopPersonalizedIndividualResponse;

class Client
{
    /**
     * @var string
     */
    protected $merchantId;

    /**
     * @var Ssl
     */
    protected $ssl;

    /**
     * @var Client
     */
    protected $client;

    /**
     * The production endpoint.
     * @var string
     */
    protected $liveEndpoint = 'https://api.blackhawknetwork.com/rewardsOrderProcessing/v1/';

    /**
     * The testing endpoint.
     * @var string
     */
    protected $testEndpoint = 'https://apipp.blackhawknetwork.com/rewardsOrderProcessing/v1/';

    /**
     * Flag to indicate if we are in test mode.
     * @var bool
     */
    protected $testMode;

    /** @var int */
    protected static $attempts = 0;

    /**
     * HawkApi constructor.
     *
     * @param string      $merchantId
     * @param Ssl         $ssl
     * @param Guzzle|null $client
     * @param bool        $testMode
     */
    public function __construct($merchantId, Ssl $ssl, Guzzle $client = null, $testMode = false)
    {
        $this->merchantId = $merchantId;
        $this->ssl = $ssl;
        $this->client = is_null($client) ? $this->getDefaultClient() : $client;
        $this->testMode = $testMode;
    }

    /**
     * Submit an giftcard individual order.
     *
     * @param  SubmitEGiftIndividualRequest $request
     * @param  string                       $requestId
     * @param  int                          $millisecondsToWait
     * @return SubmitClosedLoopAnonymousResponse|ErrorResponse
     * @throws ResponseException
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function submitEgiftIndividual(SubmitEGiftIndividualRequest $request, $requestId = null, $millisecondsToWait = 30000)
    {
        $response = $this->buildPostRequest('submitEgiftIndividual', $request, $requestId, $millisecondsToWait);

        if (in_array($response->getStatusCode(), [400, 404])) {
            return ErrorResponse::fromHttpResponse($response);
        }

        return SubmitEGiftIndividualResponse::fromHttpResponse($response);
    }

    /**
     * Submit a closed loop individual order. Giftcard.
     *
     * @param  SubmitClosedLoopIndividualRequest $request
     * @param  null $requestId
     * @param  int $millisecondsToWait
     * @return ErrorResponse|SubmitClosedLoopAnonymousResponse|static
     * @throws ResponseException
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function submitClosedLoopIndividual(SubmitClosedLoopIndividualRequest $request, $requestId = null, $millisecondsToWait = 20000)
    {
        $response = $this->buildPostRequest('submitClosedLoopIndividual', $request, $requestId, $millisecondsToWait);

        if (400 === $response->getStatusCode()) {
            return ErrorResponse::fromHttpResponse($response);
        }

        return SubmitClosedLoopIndividualResponse::fromHttpResponse($response);
    }

    /**
     * Submit a closed loop bulk order. Giftcard, multiple quantities.
     *
     * @param  SubmitClosedLoopBulkRequest $request
     * @param  null $requestId
     * @param  int $millisecondsToWait
     * @return ErrorResponse|SubmitClosedLoopBulkResponse|static
     * @throws ResponseException
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function submitClosedLoopBulk(SubmitClosedLoopBulkRequest $request, $requestId = null, $millisecondsToWait = 20000)
    {
        $response = $this->buildPostRequest('submitClosedLoopBulk', $request, $requestId, $millisecondsToWait);

        if (400 === $response->getStatusCode()) {
            return ErrorResponse::fromHttpResponse($response);
        }

        return SubmitClosedLoopBulkResponse::fromHttpResponse($response);
    }

    /**
     * Submit a closed loop order anonymously.
     *
     * @param  SubmitClosedLoopAnonymousRequest $request
     * @param  string $requestId
     * @param  int $millisecondsToWait
     *
     * @return SubmitClosedLoopAnonymousResponse|ErrorResponse
     * @throws ResponseException
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function submitClosedLoopAnonymous(SubmitClosedLoopAnonymousRequest $request, $requestId = null, $millisecondsToWait = 20000)
    {
        $response = $this->buildPostRequest('submitClosedLoopAnonymous', $request, $requestId, $millisecondsToWait);

        if (400 === $response->getStatusCode()) {
            return ErrorResponse::fromHttpResponse($response);
        }

        return SubmitClosedLoopAnonymousResponse::fromHttpResponse($response);
    }

    /**
     * Submit an open loop personalized individual (reloadable) orders.
     *
     * @param SubmitOpenLoopPersonalizedIndividualRequest $request
     * @param null $requestId
     * @param int $millisecondsToWait
     * @return SubmitOpenLoopPersonalizedIndividualResponse|ErrorResponse
     * @throws ResponseException
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function submitOpenLoopPersonalizedIndividual(SubmitOpenLoopPersonalizedIndividualRequest $request, $requestId = null, $millisecondsToWait = 20000)
    {
        $response = $this->buildPostRequest('submitOpenLoopPersonalizedIndividual', $request, $requestId, $millisecondsToWait);

        if (400 === $response->getStatusCode()) {
            return ErrorResponse::fromHttpResponse($response);
        }

        return SubmitOpenLoopPersonalizedIndividualResponse::fromHttpResponse($response);
    }

    /**
     * Submit an open loop personalized individual (reloadable) orders.
     *
     * @param SubmitFundingRequest $request
     * @param null $requestId
     * @param int $millisecondsToWait
     * @return SubmitFundingResponse|ErrorResponse
     * @throws ResponseException
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function submitFunding(SubmitFundingRequest $request, $requestId = null, $millisecondsToWait = 20000)
    {
        $response = $this->buildPostRequest('submitFunding', $request, $requestId, $millisecondsToWait);

        if (400 === $response->getStatusCode()) {
            return ErrorResponse::fromHttpResponse($response);
        }

        return SubmitFundingResponse::fromHttpResponse($response);
    }

    /**
     * Submit a request to retrieve order information.
     *
     * @param  string      $orderNumber
     * @param  string      $clientProgramNumber
     * @param  string      $orderRequestId
     * @param  string|null $requestId
     * @param  int         $millisecondsToWait
     * @return OrderInfoResponse|ErrorResponse
     * @throws ResponseException
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    public function orderInfo($orderNumber = null, $clientProgramNumber = null, $orderRequestId = null, $requestId = null, $millisecondsToWait = 20000)
    {
        $query = array_filter([
            'orderNumber' => $orderNumber,
            'clientProgramNumber' => $clientProgramNumber,
            'requestId' => $orderRequestId
        ]);

        $response = $this->buildGetRequest('orderInfo/byKeys', $query, $requestId, $millisecondsToWait);

        if (400 === $response->getStatusCode()) {
            return ErrorResponse::fromHttpResponse($response);
        }

        return OrderInfoResponse::fromHttpResponse($response);
    }

    /**
     * Build a POST request.
     *
     * @param  string $method
     * @param  Request $request
     * @param  string $requestId
     * @param  int $millisecondsToWait
     * @return \Psr\Http\Message\ResponseInterface
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    protected function buildPostRequest($method, Request $request, $requestId, $millisecondsToWait)
    {
        static::$attempts++;

        return $this->request('post', $method, [
            'cert' => $this->ssl->getCertPath(),
            'ssl_key' => $this->ssl->getKeyPath(),
            'json' => $request->toArray(),
            'headers' => $this->getDefaultHeaders([
                'RequestId' => $requestId,
                'millisecondsToWait' => $millisecondsToWait,
                'previousAttempts' => static::$attempts,
            ])
        ]);
    }

    /**
     * Build a GET request.
     *
     * @param  string $method
     * @param  array  $query
     * @param  string $requestId
     * @param  int    $millisecondsToWait
     * @return \Psr\Http\Message\ResponseInterface
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    protected function buildGetRequest($method, $query, $requestId, $millisecondsToWait)
    {
        static::$attempts++;

        return $this->request('GET', $method, [
            'cert' => $this->ssl->getCertPath(),
            'ssl_key' => $this->ssl->getKeyPath(),
            'query' => $query,
            'headers' => $this->getDefaultHeaders([
                'RequestId' => $requestId,
                'millisecondsToWait' => $millisecondsToWait,
                'previousAttempts' => static::$attempts,
            ])
        ]);
    }

    /**
     * Make a request to the API.
     *
     * @param  string $method
     * @param  string $uri
     * @param  array $options
     * @return \Psr\Http\Message\ResponseInterface
     * @throws \GuzzleHttp\Exception\GuzzleException
     */
    protected function request($method, $uri, array $options)
    {
        return $this->client->request($method, $uri, $options);
    }

    /**
     * The default headers and possibly merge in additional headers.
     *
     * @param  array $additional
     * @return array
     */
    protected function getDefaultHeaders($additional = [])
    {
        return array_merge([
            'MerchantId' => $this->merchantId, // Identifier issued by BHN [string] {required}
        ], $additional);
    }

    /**
     * The default configured HTTP client.
     *
     * @return Guzzle
     */
    protected function getDefaultClient()
    {
        return new Guzzle([
            'base_uri' => $this->testMode ? $this->liveEndpoint : $this->testEndpoint,
            'debug' => false,
            'verify' => false,
            'exceptions' => false,
            'curl' => [
                CURLOPT_SSL_VERIFYPEER => false,
                CURLOPT_SSLVERSION => CURL_SSLVERSION_TLSv1_2,
                CURLOPT_SSLCERT => $this->ssl->getCertPath(),
                CURLOPT_SSLKEY => $this->ssl->getKeyPath(),
            ]
        ]);
    }
}
