<?php

namespace Ignite\Flash;

use Illuminate\Contracts\Config\Repository;
use Illuminate\Support\HtmlString;
use Illuminate\Support\Traits\Macroable;

class Message
{
    use Macroable;

    const LEVEL_INFO = 'info';
    const LEVEL_WARNING = 'warning';
    const LEVEL_SUCCESS = 'success';
    const LEVEL_ERROR = 'error';

    /**
     * The level indicator.
     *
     * @var string
     */
    private $level;

    /**
     * The flash message.
     *
     * @var string
     */
    private $message;

    /**
     * The config repository.
     *
     * @var Repository
     */
    private $config;

    /**
     * Indicates if the message should be treated as raw HTML.
     *
     * @var bool
     */
    private $allowRawHtml = false;

    /**
     * Create a new flash message instance.
     *
     * @param string $level
     * @param string $message
     * @param Repository $config
     */
    public function __construct(string $level, string $message, Repository $config)
    {
        $this->level = $level;
        $this->message = $message;
        $this->config = $config;
    }

    /**
     * The level indicator.
     *
     * @return string
     */
    public function getLevel()
    {
        return $this->level;
    }

    /**
     * Retrieves the flash message.
     * 
     * If raw HTML is allowed for this message (via the allowHtml() method), 
     * it will return an instance of HtmlString, preventing Blade from escaping it.
     * Otherwise, it will return the message as a plain string, which Blade will escape by default.
     * 
     * WARNING: Ensure that user-generated content is never treated as raw HTML to prevent XSS vulnerabilities.
     *
     * @return string|\Illuminate\Support\HtmlString
     */
    public function getMessage()
    {
        if ($this->allowRawHtml) {
            return new HtmlString($this->message);
        }
        
        return $this->message;
    }

    /**
     * The alias for the level.
     *
     * @return string
     */
    public function getAlias()
    {
        $aliases = $this->config->get('ignite-flash.levels.aliases', []);

        if (! array_key_exists($this->level, $aliases)) {
            return $this->getLevel();
        }

        return $aliases[$this->level];
    }

    /**
     * Marks the message as safe for rendering as raw HTML.
     * 
     * WARNING: Use this method with caution. Only allow raw HTML for trusted content 
     * to avoid potential XSS vulnerabilities. Never use it with user-generated content.
     *
     * @return $this
     */
    public function allowHtml()
    {
        $this->allowRawHtml = true;
        return $this;
    }
}
