<?php

namespace Ignite\Flash;

use Exception;
use Illuminate\Support\Traits\Macroable;

class MessageHandler
{
    use Macroable;

    const SESSION_KEY = 'ignite.flash.messages';

    /**
     * The session store instance.
     *
     * @var SessionStore
     */
    protected $session;

    /**
     * The messages collection.
     *
     * @var MessageCollection
     */
    protected $collection;

    /**
     * Create instances of the Message class.
     *
     * @var MessageFactory
     */
    protected $factory;

    /**
     * MessageHandler constructor.
     *
     * @param SessionStore $session
     * @param MessageCollection $collection
     * @param MessageFactory $factory
     */
    public function __construct(SessionStore $session, MessageCollection $collection, MessageFactory $factory)
    {
        $this->session = $session;
        $this->collection = $collection;
        $this->factory = $factory;
    }

    /**
     * Flash an information message.
     *
     * @param  string $message
     * @return MessageHandler
     * @throws InvalidFlashMessageException
     */
    public function info($message)
    {
        return $this->add($message, Message::LEVEL_INFO);
    }

    /**
     * Flash a warning message.
     *
     * @param  string $message
     * @return MessageHandler
     * @throws InvalidFlashMessageException
     */
    public function warning($message)
    {
        return $this->add($message, Message::LEVEL_WARNING);
    }

    /**
     * Flash a success message.
     *
     * @param  string $message
     * @return MessageHandler
     * @throws InvalidFlashMessageException
     */
    public function success($message)
    {
        return $this->add($message, Message::LEVEL_SUCCESS);
    }

    /**
     * Flash an error message.
     *
     * @param  string $message
     * @return MessageHandler
     * @throws InvalidFlashMessageException
     */
    public function error($message)
    {
        return $this->add($message, Message::LEVEL_ERROR);
    }

    /**
     * Handle sending the flash messages to the session.
     *
     * @param  string $message
     * @param  string $level
     * @return MessageHandler
     * @throws InvalidFlashMessageException
     */
    public function add($message, $level = Message::LEVEL_INFO)
    {
        $this->guardAgainstEmptyOrInvalidMessage($message);

        $this->collection->addMessage($this->getMessageInstance($message, $level));

        $this->session->flash(static::SESSION_KEY, $this->collection);

        return $this;
    }

    /**
     * Verify a message is not empty.
     *
     * @param  string $message
     * @throws InvalidFlashMessageException
     */
    protected function guardAgainstEmptyOrInvalidMessage($message)
    {
        $message = trim($message);

        if (empty($message) || ! is_string($message)) {
            throw new InvalidFlashMessageException();
        }
    }

    /**
     * Resolve the Message instance from the container.
     *
     * @param string $message
     * @param string $level
     *
     * @return Message
     * @throws \Illuminate\Contracts\Container\BindingResolutionException
     */
    protected function getMessageInstance(string $message, string $level)
    {
        return $this->factory->make($message, $level);
    }

    /**
     * Marks the last message as safe for rendering as raw HTML.
     * 
     * WARNING: Use this method with caution. Only allow raw HTML for trusted content 
     * to avoid potential XSS vulnerabilities. Never use it with user-generated content.
     *
     * @return $this
     */
    public function allowHtml()
    {
        $message = $this->collection->last();
        $message->allowHtml();
    }

    /**
     * Marks the all messages in collection as safe for rendering as raw HTML.
     * 
     * WARNING: Use this method with caution. Only allow raw HTML for trusted content 
     * to avoid potential XSS vulnerabilities. Never use it with user-generated content.
     *
     * @return $this
     */
    public function allowHtmlAll()
    {
        $this->collection->each(function ($message) {
            $message->allowHtml();
        });
    }
}
