<?php

namespace Ignite\Flash\Test;

use Ignite\Flash\Facades\Flash;
use Ignite\Flash\Message;
use Ignite\Flash\MessageCollection;
use Ignite\Flash\MessageHandler;
use Illuminate\Support\Facades\View;

class FlashTest extends TestCase
{
    /**
     * Setup the test environment.
     *
     * @return void
     */
    public function setUp()
    {
        parent::setUp();
    }

    /**
     * Fetch the MessageCollection instance from the session.
     *
     * @param string $key
     *
     * @return MessageCollection
     */
    private function collectionFromSession($key = MessageHandler::SESSION_KEY)
    {
        return $this->app->session->get($key);
    }

    /**
     * Assert that the session has the MessageCollection instance under the expected key.
     *
     * @param string $message
     * @param string $key
     */
    private function assertSessionHasMessageCollection($message = '', $key = MessageHandler::SESSION_KEY)
    {
        static::assertThat($this->app->session->has($key), static::isTrue(), $message);
    }

    /**
     * Simple data provider.
     *
     * @return array
     */
    public function methodProvider()
    {
        return [
            ['add', 'foo', 'bar'],                                  // => Flash::add('foo', 'bar);
            ['info', 'Info message', Message::LEVEL_INFO],          // => Flash::info('Info message');
            ['error', 'Error message', Message::LEVEL_ERROR],       // => Flash::error('Error message');
            ['warning', 'Warning message', Message::LEVEL_WARNING], // => Flash::warning('Warning message');
            ['success', 'Success message', Message::LEVEL_SUCCESS], // => Flash::success('Success message');
        ];
    }

    /**
     * @test
     * @dataProvider methodProvider
     *
     * @param string $method
     * @param string $message
     * @param string $level
     */
    public function it_can_flash_an_arbitrary_message($method, $message, $level)
    {
        forward_static_call_array(['Flash', $method], compact('message', 'level'));

        $this->assertSessionHasMessageCollection();
        $collection = $this->collectionFromSession();
        $this->assertCount(1, $collection);
        $this->assertEquals($level, $collection->first()->getLevel());
        $this->assertEquals($message, $collection->first()->getMessage());
    }

    /**
     * @test
     */
    public function it_can_flash_multiple_messages()
    {
        Flash::info($message1 = 'Your message was queued for delivery.');
        Flash::warning($message2 = 'You are now logged in as John.');
        Flash::error($message3 = 'You cannot log in as John.');
        Flash::success($message4 = 'Your message was delivered.');

        $collection = $this->collectionFromSession();
        $this->assertCount(4, $collection);
        $this->assertEquals(Message::LEVEL_INFO, $collection->get(0)->getLevel());
        $this->assertEquals($message1, $collection->get(0)->getMessage());
        $this->assertEquals(Message::LEVEL_WARNING, $collection->get(1)->getLevel());
        $this->assertEquals($message2, $collection->get(1)->getMessage());
        $this->assertEquals(Message::LEVEL_ERROR, $collection->get(2)->getLevel());
        $this->assertEquals($message3, $collection->get(2)->getMessage());
        $this->assertEquals(Message::LEVEL_SUCCESS, $collection->get(3)->getLevel());
        $this->assertEquals($message4, $collection->get(3)->getMessage());
    }

    /**
     * @test
     */
    public function it_can_render_the_flash_messages_in_the_view()
    {
        Flash::info($message1 = 'Your message was queued for delivery.');
        Flash::error($message2 = 'You cannot log in as John.');
        Flash::warning($message3 = 'You are now logged in as John.');
        Flash::success($message4 = 'Your message was delivered.');

        $view = View::make('ignite-flash::message')->render();

        $this->assertTrue((bool) preg_match('/alert-info/', $view), 'Assert alert-info is in the view');
        $this->assertTrue((bool) preg_match("/{$message1}/", $view), "Assert '{$message1}' is in the view");

        $this->assertTrue((bool) preg_match('/alert-danger/', $view), 'Assert alert-danger is in the view');
        $this->assertTrue((bool) preg_match("/{$message2}/", $view), "Assert '{$message2}' is in the view");

        $this->assertTrue((bool) preg_match('/alert-warning/', $view), 'Assert alert-warning is in the view');
        $this->assertTrue((bool) preg_match("/{$message3}/", $view), "Assert '{$message3}' is in the view");

        $this->assertTrue((bool) preg_match('/alert-success/', $view), 'Assert alert-success is in the view');
        $this->assertTrue((bool) preg_match("/{$message4}/", $view), "Assert '{$message4}' is in the view");
    }
}
