<?php

namespace Ignite\Flash;

use Illuminate\Contracts\Session\Session;

class MessageHandler
{
    const LEVEL_INFO = 'info';
    const LEVEL_WARNING = 'warning';
    const LEVEL_SUCCESS = 'success';
    const LEVEL_ERROR = 'danger';

    /** @var Session */
    protected $session;

    /**
     * MessageHandler constructor.
     *
     * @param Session $session
     */
    public function __construct(Session $session)
    {
        $this->session = $session;
    }

    /**
     * Flash an information message.
     *
     * @param  string $message
     * @return MessageHandler
     * @throws \Exception
     */
    public function info($message)
    {
        return $this->handle($message, static::LEVEL_INFO);
    }

    /**
     * Flash a warning message.
     *
     * @param  string $message
     * @return MessageHandler
     * @throws \Exception
     */
    public function warning($message)
    {
        return $this->handle($message, static::LEVEL_INFO);
    }

    /**
     * Flash a success message.
     *
     * @param  string $message
     * @return MessageHandler
     * @throws \Exception
     */
    public function success($message)
    {
        return $this->handle($message, static::LEVEL_SUCCESS);
    }

    /**
     * Flash an error message.
     *
     * @param  string $message
     * @return MessageHandler
     * @throws \Exception
     */
    public function error($message)
    {
        return $this->handle($message, static::LEVEL_ERROR);
    }

    /**
     * Flash a custom message.
     *
     * @param  string $message
     * @param  string $type
     * @return MessageHandler
     * @throws \Exception
     */
    public function message($message, $type)
    {
        return $this->handle($message, $type);
    }

    /**
     * Handle sending the flash message to the session.
     *
     * @param  string $message
     * @param  string $level
     * @return $this
     * @throws \Exception
     */
    public function handle($message, $level = self::LEVEL_INFO)
    {
        $this->verify($message);

        $this->session->put('ignite.flash.message', $message);
        $this->session->put('ignite.flash.level', $level);

        return $this;
    }

    /**
     * Verify a message is not empty.
     *
     * @param  string $message
     * @throws \Exception
     */
    protected function verify($message)
    {
        if (empty($message)) {
            throw new InvalidFlashMessageException();
        }
    }
}
