<?php

namespace Ignite\Demo\Generators;

use Closure;
use Ignite\Core\Entities\User;
use Ignite\Core\Entities\Participant;
use Ignite\Core\Contracts\Repositories\ParticipantRepository;
use Ignite\Demo\Generator;
use Ignite\Demo\Traits\ParticipantTypesTrait;

class Participants
{
    use ParticipantTypesTrait;

    /** @var array */
    protected $users = [
        ["Hailey", "Thomas", "6019"],
        ["Kim", "Fox", "6036"],
        ["Cheri", "Lacoss", "6054"],
        ["Kim", "Dierks", "6065"],
        ["Holly", "Swanzy", "6031"],
        ["Nicole", "Chattin", "6034"],
        ["Genny", "Castleberry", "6020"],
        ["Mike", "May", "6010"],
        ["Rudy", "Garza", "6070"],
        ["Brie", "Richards", "6025"],
        ["Allie", "Greenwood", "6023"],
        ["Courtney", "Neunaber", "6011"],
        ["Phoebe", "Wright", "6053"],
        ["Karolyn", "Hemmig", "6061"],
        ["Julie", "Blank", "6042"],
        ["Michael", "Butler", "6040"],
        ["Kelly", "Kunz", "6052"],
        ["Jason", "Neal", "6041"],
        ["Michael", "Pisterzi", "6081"],
        ["Huy", "Nguyen", "6043"],
        ["Eoghan", "O'Brien", "6031"],
        ["Caroline", "Van Veldhuizen", "6056"],
        ["Marisela", "Moreno", "6043"],
        ["Taylor", "Renner", "6021"],
        ["Courtney", "Schroeder", "6046"],
        ["Pali", "Chhabra", "6058"],
    ];

    /** @var ParticipantRepository */
    protected $participantRepository;

    /**
     * Participants constructor.
     *
     * @param ParticipantRepository $participantRepository
     */
    public function __construct(ParticipantRepository $participantRepository)
    {
        $this->participantRepository = $participantRepository;
    }

    /**
     * Handle the generation of demo data.
     *
     * @param  Generator $generator
     * @param  Closure   $next
     * @return mixed
     */
    public function handle(Generator $generator, Closure $next)
    {
        foreach ($this->users as $user) {
            $email = $this->getIdentifier($user);
            if ($email === config('core.admin.email')) {
                continue;
            }

            $exists = (int) Participant::query()->where('email', $email)->count();

            if ($exists) {
                $generator->addError($this, "Participant already exists: {$email}");
                continue;
            }

            try {
                $participant = $this->createParticipant($user);
                $participant->user->assignToGroup('Participant');
            } catch (\Exception $e) {
                $generator->addError($this, $e);
                continue;
            }
        }

        return $next($generator);
    }

    /**
     * Create a participant record.
     *
     * @param  array $user
     * @return Participant
     * @throws \Exception
     */
    protected function createParticipant(array $user)
    {
        return $this->participantRepository->create(
            $this->getParticipantData($user)
        );
    }

    /**
     * Get the necessary participant data.
     *
     * @param  array $user
     * @return array
     */
    protected function getParticipantData($user)
    {
        $email = $this->getIdentifier($user);
        $date = now()->subDays(rand(1, 100));

        return [
            'user' => [
                'password' => bcrypt('br1ght1d34s!'),
                'status' => User::STAT_ACTIVE
            ],
            'participant' => [
                'first' => $user[0],
                'last' => $user[1],
                'email' => $email,
                'type' => $this->getRandomType(),
                'company' => 'Brightspot',
                'phone1' => '972.661.' . $user[2],
                'address_1' => '102 Decker Court',
                'address_2' => 'Suite 150',
                'city' => 'Irving',
                'state' => 'TX',
                'postal' => '75062',
                'country' => 'US',
                'approved_at' => $date,
                'created_at' => $date
            ]
        ];
    }

    /**
     * Get the identifier for the record.
     *
     * @param  array $user
     * @return string
     */
    protected function getIdentifier($user)
    {
        $user = array_map('strtolower', $user);
        $email = sprintf('%s.%s@brightspot.email', $user[0], $user[1]);

        return str_replace([' ', "'"], '', $email);
    }
}
