<?php

namespace Ignite\Demo\Generators;

use Closure;
use Illuminate\Support\Collection;
use Ignite\Catalog\Contracts\FavoriteRepository;
use Ignite\Catalog\Contracts\OrderRepository;
use Ignite\Catalog\Entities\CartItem;
use Ignite\Catalog\Entities\Item;
use Ignite\Catalog\Models\Cart;
use Ignite\Catalog\Models\Product\BuyRequest;
use Ignite\Core\Entities\Participant;
use Ignite\Core\Entities\Transaction;
use Ignite\Core\Repositories\TransactionRepository;
use Ignite\Demo\Generator;

class Orders
{
    /** @var OrderRepository */
    protected $orderRepository;

    /** @var FavoriteRepository */
    protected $favoriteRepository;

    /**
     * claims constructor.
     *
     * @param OrderRepository $orderRepository
     * @param FavoriteRepository $favoriteRepository
     */
    public function __construct(
        OrderRepository $orderRepository,
        FavoriteRepository $favoriteRepository
    ) {
        $this->orderRepository = $orderRepository;
        $this->favoriteRepository = $favoriteRepository;
    }

    /**
     * Handle generation of the demo data.
     *
     * @param  Generator $generator
     * @param  Closure $next
     * @return Generator
     */
    public function handle(Generator $generator, Closure $next)
    {
        $participants = Participant::with('user')->get();

        try {
            foreach ($participants as $participant) {
                $this->createOrdersForParticipant($participant);
            }
        } catch (\Exception $e) {
            $generator->addError($this, $e);
            return $generator;
        }

        return $next($generator);
    }

    /**
     * Create orders for participant.
     *
     * @param  Participant $participant
     * @throws \Exception
     */
    protected function createOrdersForParticipant(Participant $participant)
    {
        auth()->login($participant->user);
        app('authorization')->forgetPermissions()->registerPermissions();

        $data = $this->getOrderDataFromParticipant($participant);
        $this->setStartingBalance($participant, 100000);


        if ((int) $participant->user_id !== 1) {
            $items = $this->getRandomItems();
            $balance = app(TransactionRepository::class)->getBalance();
            $total = $items->sum('points');
            $cart = new Cart($items, $balance, $total);

            $data['quantity'] = $items->count();
            $date = now()->subDays(rand(0, 60));
            $data['created_at'] = $date;
            $data['updated_at'] = $date;

            $this->orderRepository->create($data, $cart->items(), $cart->total());

            //$orders = $this->orderRepository->findAllByUser($participant->user_id);
            //$this->orderRepository->process($orders->first()->number);
            //$this->orderRepository->cancel($orders->last()->number);

            $this->favoriteRepository->favorite($this->getRandomItem(), $participant->user()->first());
        }
    }

    /**
     * Get a random item from the database.
     *
     * @return Item
     */
    protected function getRandomItem()
    {
        return Item::query()
                   ->where('type', 'simple')
                   ->where('active', 1)
                   ->take(1)
                   ->inRandomOrder()
                   ->first();
    }

    /**
     * Set a starting balance for the participant.
     *
     * @param Participant $participant
     * @param int $points
     */
    protected function setStartingBalance(Participant $participant, int $points)
    {
        Transaction::create([
            'user_id' => $participant->user_id,
            'related_id' => $participant->user_id,
            'related_name' => 'PARTICIPANT',
            'related_type' => Participant::class,
            'description' => 'Demo points',
            'type' => 'EARNED',
            'value' => $points,
            'tax_date' => now(),
            'transaction_date' => now(),
        ]);
    }

    /**
     * Get the necessary order data from the participant.
     *
     * @param  Participant $participant
     * @return array
     */
    protected function getOrderDataFromParticipant(Participant $participant)
    {
        return [
            'user_id' => $participant->user_id,
            'ship_name' => $participant->fullName(),
            'ship_email' => $participant->email,
            'ship_phone' => $participant->phone1,
            'ship_address_1' => $participant->address_1,
            'ship_address_2' => $participant->address_2,
            'ship_address_3' => '',
            'ship_city' => $participant->city,
            'ship_state' => $participant->state,
            'ship_postal' => $participant->postal,
            'ship_country' => $participant->country,
        ];
    }

    /**
     * Get a collection of random items.
     *
     * @return Collection
     */
    protected function getRandomItems()
    {
        return Item::query()->where('type', 'simple')->where('active', 1)->take(3)->inRandomOrder()->get()->map(function ($item) {
            return new BuyRequest($item->getKey(), 1);
        })->map(function (BuyRequest $request) {
            return new CartItem($request->toArray());
        });
    }
}
