<?php

namespace Ignite\Demo\Console;

use Illuminate\Console\Command;
use Ignite\Demo\Generator;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Input\InputArgument;

class Demo extends Command
{
    const INSTALL = 'install';
    const INSTALL_FRESH = 'install-fresh';

    /**
     * The console command name.
     *
     * @var string
     */
    protected $name = 'ignite:demo';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Install the demo data.';

    /**
     * Create a new command instance.
     */
    public function __construct()
    {
        parent::__construct();
    }

    /**
     * Execute the console command.
     *
     * @return mixed
     */
    public function handle()
    {
        $action = $this->input->getArgument('action');

        \Auth::loginUsingId(1);

        if ($action === self::INSTALL_FRESH) {
            $this->call('migrate:refresh');
            $this->call('module:seed');
            return $this->install();
        }

        if ($action === self::INSTALL) {
            return $this->install();
        }

        throw new \InvalidArgumentException("Action '$action' is not supported.");
    }

    /**
     * The basic outline of the demo data install is a follows:
     *
     * - Generate 50 users/participants
     * - Generate 4 offers
     * - Generate 1 rule
     * - Generate at least 3 claims per participant
     *  - Attach random participants
     *  - Attach random line items
     *
     * @return bool
     */
    protected function install()
    {
        $this->output->writeln("<info>Starting demo data installation...</info>");

        $installClaimsData = (bool) $this->option('claims');
        $installOrdersData = (bool) $this->option('orders');

        if ($installOrdersData) {
            $this->call('ignite:catalog:hawk', [
                'product' => 93365701,
                '--markup' => 0,
                '--point' => 1,
                '--denominations' => '10,25,50,100'
            ]);
        }

        $generator = $this->generator()->run($installClaimsData, $installOrdersData)->then(function($generator) {
            return $generator;
        });

        if ($generator->errors()->isNotEmpty()) {
            $this->renderErrors($generator->errors()->toArray());
        }

        $this->output->writeln("<info>Finished generating demo data.</info>");

        return $generator->errors()->isEmpty();
    }

    /**
     * Render the errors to the console.
     *
     * @param array $errors
     */
    protected function renderErrors($errors)
    {
        $this->output->writeln("<error>There were errors while generating the demo data:</error>");
        foreach ($errors as $class => $error) {
            $this->output->writeln("<error>$class:</error>");
            foreach ($error as $err) {
                $this->output->writeln("<error>$err</error>");
            }
        }
    }


    /**
     * Get the console command arguments.
     *
     * @return array
     */
    protected function getArguments()
    {
        return [
            ['action', InputArgument::REQUIRED, "'install' to just install the data.\n'install-fresh' to re-migrate, seed and install the data."],
        ];
    }

    /**
     * Get the console command options.
     *
     * @return array
     */
    protected function getOptions()
    {
        return [
            ['claims', null, InputOption::VALUE_OPTIONAL, 'Only claim data', false],
            ['orders', null, InputOption::VALUE_OPTIONAL, 'Only order data', false]
        ];
    }

    /**
     * The generator instance.
     *
     * @return Generator
     */
    protected function generator()
    {
        return app(Generator::class);
    }
}
