<?php

namespace Ignite\Course\Http\Controllers\Admin;

use Exception;
use Ignite\Activity\Entities\Type;
use Ignite\Core\Http\Controllers\Controller;
use Ignite\Core\Models\Grid\GenericHistoryTable;
use Ignite\Course\Entities\Course;
use Ignite\Course\Entities\CourseFamily;
use Ignite\Course\Http\Forms\CourseForm;
use Ignite\Course\Models\Grid\CourseTable;
use Ignite\Flash\MessageHandler;
use Illuminate\Http\Request;

class CourseController extends Controller
{
    /**
     * Show the admin index page which will list Course by status.
     *
     * @param CourseTable $table
     * @param Request      $request
     * @return JsonResponse|View
     */
    public function index(Type $type, CourseFamily $courseFamily, CourseTable $table, Request $request)
    {
        $table->setCourseFamily($courseFamily);
        if ($request->wantsJson()) {
            return $table->ajax();
        }

        return $table->render('Course::admin.course.index', compact('type', 'courseFamily'));
    }

    /**
     * Show the Course view page.
     *
     * @param Type $type
     * @param Course $course
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function show(Type $type, Course $course)
    {
        $model = $course;
        $form = $this->form(CourseForm::class, [
            'model' => $model,
        ], [
            'readonly' => true
        ]);

        if (!session()->has(MessageHandler::SESSION_KEY)) {
            $this->flashPathStatus($model);
            // view flash on current page; do not show on next page
            session()->ageFlashData();
        }

        return view('Course::admin.course.show', compact('type', 'model', 'form'));
    }

    /**
     * Create a Course.
     *
     * @param Type $type
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function create(Type $type, CourseFamily $courseFamily)
    {
        $form = $this->form(CourseForm::class, [
            'method' => 'POST',
            'url' => route('admin.course.store', ['type' => $type, 'courseFamily' => $courseFamily]),
        ]);

        return view('Course::admin.course.create', compact('type', 'courseFamily', 'form'));
    }

    /**
     * Save the new Course.
     *
     * @param Type
     * @param CourseFamily
     * @return RedirectResponse
     */
    public function store(Type $type, CourseFamily $courseFamily)
    {
        $form = $this->form(CourseForm::class);

        $form->redirectIfNotValid();

        try {
            $data = $form->getFieldValues();
            $data['activity_course_family_id'] = $courseFamily->id;
            $upload = $data['upload'];
            unset($data['upload']);

            if (!isset($data['status'])) {
                $data['status'] = Course::PENDING;
            }

            $model = Course::create($data);
            if ($upload) {
                $model->updatePath($upload);
            }

            $this->flash('success', __("Course::types.{$type->code}.Course") . ' created successfully.');
            $this->flashPathStatus($model);

            return redirect()->route('admin.course.show', ['type' => $type->code, 'course' => $model]);
        } catch (Exception $e) {
            $this->logException($e);
            $this->flash(
                'error',
                'We were unable to create the '
                . __("Course::types.{$type->code}.Course")
                . '. Please contact IT: ' . $e->getMessage()
            );
            return redirect()->back();
        }
    }

    /**
     * Show the Course edit page.
     *
     * @param Type $type
     * @param Course $course
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function edit(Type $type, Course $course)
    {
        $model = $course;
        $form = $this->form(CourseForm::class, [
            'method' => 'POST',
            'url' => route(
                'admin.course.update',
                [
                    'type' => $type,
                    'course' => $model,
                ]
            ),
            'model' => $model,
        ]);

        if (!session()->has(MessageHandler::SESSION_KEY)) {
            $this->flashPathStatus($model);
            // view flash on current page; do not show on next page
            session()->ageFlashData();
        }

        return view('Course::admin.course.edit', compact('type', 'form', 'model'));
    }

    /**
     * Show the audit history related to the Course .
     *
     * @param Type $type
     * @param  Course $course
     * @param  Request $request
     * @return \Illuminate\Http\JsonResponse|\Illuminate\View\View
     */
    public function history(Type $type, Course $course, Request $request)
    {
        $model = $course;
        $table = resolve(GenericHistoryTable::class)->setModel($model);
        if ($request->wantsJson()) {
            return $table->ajax();
        }

        return $table->render('Course::admin.course.history', compact('type', 'model'));
    }

    /**
     * Save the Course edit page.
     *
     * @param Type $type
     * @param Course $course
     * @return RedirectResponse
     */
    public function update(Type $type, Course $course)
    {
        $model = $course;
        $form = $this->form(CourseForm::class, [
            'model' => $model,
        ]);

        $form->redirectIfNotValid();

        try {
            $data = $form->getFieldValues(true);
            $upload = $data['upload'];
            unset($data['upload']);

            $model->update($data);
            if ($upload) {
                $model->updatePath($upload);
            }

            $this->flash('success', __("Course::types.{$type->code}.Course") . ' updated successfully.');
            $this->flashPathStatus($model);

            return redirect()->route('admin.course.show', ['type' => $type->code, 'course' => $model]);
        } catch (Exception $e) {
            $this->logException($e);
            $this->flash(
                'error',
                'We were unable to update the '
                . __("Course::types.{$type->code}.Course")
                . '. Please contact IT: ' . $e->getMessage()
            );
            return redirect()->back();
        }
    }

    /**
     * Flash the status of the path if any.
     *
     * @param  Course $course
     */
    protected function flashPathStatus(Course $course)
    {
        $message = $course->getPathStatusMessage();
        if ($message) {
            $status = $course->getPathStatus();
            $this->flash(
                Course::UPLOADSTATUS_ERROR == $status ? 'error' : 'info',
                Course::UPLOADSTATUS_ERROR == $status ? 'Upload Error: ' . $message : $message,
            );
        }
    }
}
