<?php

namespace Ignite\Course\Models\Grid;

use Ignite\Core\Models\Grid\Formatter;
use Ignite\Core\Models\Grid\QueryTable;
use Ignite\Core\Models\Menu\ActionMenu;
use Ignite\Course\Entities\Course;
use Ignite\Course\Entities\CourseFamily;

class CourseTable extends QueryTable
{
    /**
     * @var Formatter
     */
    protected $formatter = null;

    /**
     * @var CourseFamily
     */
    protected CourseFamily $courseFamily;

    /**
     * @param CourseFamily $courseFamily
     */
    public function setCourseFamily(CourseFamily $courseFamily)
    {
        $this->courseFamily = $courseFamily;
    }

    /**
     * The columns to show.
     *
     * @return array
     */
    public function columns()
    {
        $type = config('course.default_type');
        $fields = config('course.types.' . $type . '.course_fields');

        unset($fields['upload']);
        unset($fields['meta']);

        return array_merge(
            ['actions' => $this->actionsColumn()],
            $fields
        );
    }

    /**
     * The participants query.
     *
     * @return \Illuminate\Database\Query\Builder
     */
    public function query()
    {
        $columns = $this->getColumnNames(['actions'])->toArray();
        $query = Course::getQuery()->select($columns)
            ->where('activity_course_family_id', $this->courseFamily->id);

        return $query;
    }

    /**
     * Custom column for actions.
     *
     * @param  Course $model
     * @return \Illuminate\Support\HtmlString
     */
    protected function actions($model)
    {
        $type = config('course.default_type');
        $manager = resolve(ActionMenu::class);
        $manager->merge(
            $manager->hydrateItems(
                collect([
                    [
                        'position' => 10,
                        'url' => route(
                            'admin.course.show',
                            ['type' => $type, 'course' => $model->id]
                        ),
                        'label' => 'View',
                        'icon' => 'eye-open',
                        'allow' => auth()->user()->can('activity.course.all.view'),
                        'children' => [
                            [
                                'position' => 10,
                                'url' => route(
                                    'admin.course.edit',
                                    ['type' => $type, 'course' => $model->id]
                                ),
                                'label' => 'Edit',
                                'icon' => 'edit',
                                'allow' => auth()->user()->can('activity.course.all.update'),
                            ],
                            [
                                'position' => 20,
                                'url' => route(
                                    'admin.course.history',
                                    ['type' => $type, 'course' => $model->id]
                                ),
                                'label' => 'History',
                                'icon' => 'history',
                                'allow' => auth()->user()->can('activity.course.all.view'),
                            ],
                        ]
                    ],
                ])
            )
        );

        return $this->formatter()->html(
            $manager->render([
                'primaryKey' => $model->id,
                'model' => CourseFamily::class
            ])
        );
    }

    /**
     * @return GenericFormatter
     */
    protected function formatter(): Formatter
    {
        if (null === $this->formatter) {
            $this->formatter = resolve(Formatter::class);
        }

        return $this->formatter;
    }

    /**
     * Get a map of column keys => functions to format columns.
     *
     * @return array
     */
    protected function getColumnFormattingMap()
    {
        return [
            'actions' => function ($model) {
                return $this->actions($model);
            },
            'description' => function ($model) {
                return substr($model->description, 0, 50)
                    . (strlen($model->description) > 50 ? '...' : '');
            },
            'locale' => function ($model) {
                return strtoupper($model->locale);
            },
            'status' => function ($model) {
                return strtoupper($model->status);
            },
        ];
    }

    /**
     * Get default builder parameters.
     *
     * @return array
     */
    protected function getBuilderParameters1()
    {
        $params = [
            'order' => [
                [2, 'asc'],
            ],
        ];

        return parent::getBuilderParameters(
            $params,
            ...func_get_args()
        );
    }
}
