<?php

namespace Ignite\Course\Http\Controllers\Admin;

use Exception;
use Ignite\Activity\Entities\Offer;
use Ignite\Activity\Entities\Type;
use Ignite\Core\Http\Controllers\Controller;
use Ignite\Core\Models\Grid\GenericHistoryTable;
use Ignite\Course\Entities\CourseFamily;
use Ignite\Course\Http\Forms\CourseFamilyForm;
use Ignite\Course\Models\Grid\CourseFamilyTable;
use Illuminate\Http\Request;

class CourseFamilyController extends Controller
{
    /**
     * Show the admin index page which will list CourseFamily by status.
     *
     * @param Type $type
     * @param CourseFamilyTable $table
     * @param Request $request
     * @return JsonResponse|View
     */
    public function index(Type $type, CourseFamilyTable $table, Request $request)
    {
        if ($request->wantsJson()) {
            return $table->ajax();
        }

        return $table->render('Course::admin.course_family.index', compact('type'));
    }

    /**
     * Show the CourseFamily view page.
     *
     * @param Type $type
     * @param CourseFamily $courseFamily
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function show(Type $type, CourseFamily $courseFamily)
    {
        $model = $courseFamily;
        $form = $this->form(CourseFamilyForm::class, [
            'model' => $model,
        ], [
            'readonly' => true
        ]);

        return view('Course::admin.course_family.show', compact('type', 'model', 'form'));
    }

    /**
     * Create a CourseFamily.
     *
     * @param Type $type
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function create(Type $type)
    {
        $form = $this->form(CourseFamilyForm::class, [
            'method' => 'POST',
            'url' => route('admin.course_family.store', ['type' => $type]),
        ]);

        return view('Course::admin.course_family.create', compact('type', 'form'));
    }

    /**
     * Save the new CourseFamily.
     *
     * @param Type
     * @return RedirectResponse
     */
    public function store(Type $type)
    {
        $form = $this->form(CourseFamilyForm::class);

        $form->redirectIfNotValid();

        try {
            $data = $form->getFieldValues();
            if (!isset($data['offer_id'])) {
                $offers = Offer::where('type_id', $type->id)->get();
                if ($offers->count() > 1) {
                    throw new Exception('More than one offer found for this course type for the default.');
                }
                if ($offers->count() == 0) {
                    throw new Exception('No offer found for this course type for the default.');
                }
                $data['offer_id'] = $offers->first()->id;
            }
            $model = CourseFamily::create($data);

            $this->flash('success', __("Course::types.{$type->code}.Course Family") . ' created successfully.');
            return redirect()->route('admin.course_family.show', ['type' => $type->code, 'courseFamily' => $model]);
        } catch (Exception $e) {
            $this->logException($e);
            $this->flash(
                'error',
                'We were unable to create the '
                . __("Course::types.{$type->code}.Course Family")
                . '. Please contact IT: ' . $e->getMessage()
            );
            return redirect()->back();
        }
    }

    /**
     * Show the CourseFamily edit page.
     *
     * @param Type $type
     * @param CourseFamily $courseFamily
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function edit(Type $type, CourseFamily $courseFamily)
    {
        $model = $courseFamily;
        $form = $this->form(CourseFamilyForm::class, [
            'method' => 'POST',
            'url' => route(
                'admin.course_family.update',
                [
                    'type' => $type,
                    'courseFamily' => $model,
                ]
            ),
            'model' => $model,
        ]);

        return view('Course::admin.course_family.edit', compact('type', 'form', 'model'));
    }

    /**
     * Show the audit history related to the CourseFamily .
     *
     * @param Type $type
     * @param  CourseFamily $courseFamily
     * @param  Request $request
     * @return \Illuminate\Http\JsonResponse|\Illuminate\View\View
     */
    public function history(Type $type, CourseFamily $courseFamily, Request $request)
    {
        $model = $courseFamily;
        $table = resolve(GenericHistoryTable::class)->setModel($model);
        if ($request->wantsJson()) {
            return $table->ajax();
        }

        return $table->render('Course::admin.course_family.history', compact('type', 'model'));
    }

    /**
     * Save the CourseFamily edit page.
     *
     * @param Type $type
     * @param CourseFamily $courseFamily
     * @return RedirectResponse
     */
    public function update(Type $type, CourseFamily $courseFamily)
    {
        $model = $courseFamily;
        $form = $this->form(CourseFamilyForm::class, [
            'model' => $model,
        ]);

        $form->redirectIfNotValid();

        try {
            $data = $form->getFieldValues(true);
            $model->update($data);
            $this->flash('success', __("Course::types.{$type->code}.Course Family") . ' updated successfully.');
            return redirect()->route('admin.course_family.show', ['type' => $type->code, 'courseFamily' => $model]);
        } catch (Exception $e) {
            $this->logException($e);
            $this->flash(
                'error',
                'We were unable to update the '
                . __("Course::types.{$type->code}.Course Family")
                . '. Please contact IT: ' . $e->getMessage()
            );
            return redirect()->back();
        }
    }
}
