<?php

namespace Ignite\Course\Entities;

use Ignite\Activity\Entities\Offer;
use Ignite\Core\Entities\Base;
use Ignite\Course\Entities\Course;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Carbon;
use OwenIt\Auditing\Auditable;
use OwenIt\Auditing\Contracts\Auditable as AuditableContract;

/**
 * Ignite\Course\Entities\CourseFamily
 *
 * @property int $id
 * @property int|null $offer_id
 * @property string|null $name
 * @property string|null $category
 * @property string|null $image
 * @property string|null $passing_score
 * @property int|null $points
 * @property Carbon|null $publish_start
 * @property Carbon|null $publish_end
 * @property int $num_active
 * @property int|null $sequence
 * @property string|null $notes
 * @property Carbon|null $deleted_at
 * @property Carbon|null $created_at
 * @property Carbon|null $updated_at
 * @property-read \Illuminate\Database\Eloquent\Collection|\Ignite\Core\Entities\Audit[] $audits
 * @property-read int|null $audits_count
 * @property-read \Illuminate\Database\Eloquent\Collection|\Ignite\Course\Entities\CourseCompletion[] $completions
 * @property-read int|null $completions_count
 * @property-read \Illuminate\Database\Eloquent\Collection|Course[] $courses
 * @property-read int|null $courses_count
 * @property-read \Illuminate\Database\Eloquent\Collection|Course[] $locales
 * @property-read int|null $locales_count
 * @property-read Offer|null $offer
 * @method static Builder|CourseFamily forCategory(string $category)
 * @method static Builder|CourseFamily forOffer(\Ignite\Activity\Entities\Offer $offer)
 * @method static Builder|CourseFamily newModelQuery()
 * @method static Builder|CourseFamily newQuery()
 * @method static Builder|CourseFamily onlyActive()
 * @method static \Illuminate\Database\Query\Builder|CourseFamily onlyTrashed()
 * @method static Builder|CourseFamily query()
 * @method static Builder|CourseFamily whereCategory($value)
 * @method static Builder|CourseFamily whereCreatedAt($value)
 * @method static Builder|CourseFamily whereDeletedAt($value)
 * @method static Builder|Base whereHasPermission(string $permission, ?\Ignite\Core\Entities\User $user = null)
 * @method static Builder|CourseFamily whereId($value)
 * @method static Builder|CourseFamily whereImage($value)
 * @method static Builder|CourseFamily whereName($value)
 * @method static Builder|CourseFamily whereNotes($value)
 * @method static Builder|CourseFamily whereNumActive($value)
 * @method static Builder|CourseFamily whereOfferId($value)
 * @method static Builder|CourseFamily wherePassingScore($value)
 * @method static Builder|CourseFamily wherePoints($value)
 * @method static Builder|CourseFamily wherePublishEnd($value)
 * @method static Builder|CourseFamily wherePublishStart($value)
 * @method static Builder|CourseFamily whereSequence($value)
 * @method static Builder|CourseFamily whereUpdatedAt($value)
 * @method static \Illuminate\Database\Query\Builder|CourseFamily withTrashed()
 * @method static \Illuminate\Database\Query\Builder|CourseFamily withoutTrashed()
 * @mixin \Eloquent
 */
class CourseFamily extends Base implements AuditableContract
{
    use Auditable;
    use SoftDeletes;

    /**
     * The table name.
     * @var string
     */
    protected $table = 'activity_course_family';

    /**
     * Should the timestamps be audited?
     *
     * @var bool
     */
    protected $auditTimestamps = false;

    /**
     * Custom Audit Driver
     *
     * @var \Ignite\Core\Audit\ParticipantDriver
     */
    protected $auditDriver = \Ignite\Core\Audit\GenericDriver::class;

    /**
     * Attributes to exclude from the Audit.
     *
     * @var array
     */
    protected $auditExclude = [];

    /**
     * The attributes that should be casted.
     *
     * @var array
     */
    protected $casts = [
        'publish_start' => 'date',
        'publish_end' => 'date',
        'deleted_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * The identifier for the record in the audit log.
     *
     * @return mixed
     */
    public static function getAuditFriendlyField()
    {
        return ['name'];
    }

    /**
     * The relationship to the completions.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function completions(): HasMany
    {
        return $this->hasMany(CourseCompletion::class, $this->table . '_id');
    }

    /**
     * The relationship to the courses.
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function courses(): HasMany
    {
        return $this->hasMany(Course::class, $this->table . '_id');
    }

    /**
     * Returns if the course is active.
     *
     * @return boolean
     */
    public function isActive(): bool
    {
        return $this->num_active > 0;
    }

    /**
     * Is the course currently published.
     *
     * @param  Carbon $date
     * @return boolean
     */
    public function isCurrentlyPublsh(Carbon $date = null): bool
    {
        if (!$date) {
            $date = now();
        }

        if ($this->publish_start) {
            if ($this->publish_end) {
                return $this->publish_start <= $date && $this->publish_end > $date;
            } else {
                return $this->publish_start <= $date;
            }
        } elseif ($this->publish_end) {
            return $this->publish_end > $date;
        }

        return false;
    }

    /**
     * Returns if the course is active.
     *
     * @return boolean
     */
    public function isInactive(): bool
    {
        return $this->num_active == 0;
    }

    /**
     * Alias for course()
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function locales(): HasMany
    {
        return $this->courses();
    }

    /**
     * The relationship to the offer.
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function offer(): BelongsTo
    {
        return $this->belongsTo(Offer::class);
    }

    /**
     * Scope a query to filter by category.
     *
     * @param  Builder $query
     * @param  string  $category
     * @return void
     */
    public function scopeForCategory(Builder $query, string $category): void
    {
        $query->where('category', $category);
    }

    /**
     * Scope a query to filter by offer.
     *
     * @param  Builder $query
     * @param  Offer $offer
     * @return void
     */
    public function scopeForOffer(Builder $query, Offer $offer): void
    {
        $query->where('offer_id', $offer->id);
    }

    /**
     * Only active courses.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     */
    public function scopeOnlyActive($query): void
    {
        $query->where('num_active', '>', 0);
    }

    /**
     * Updates the number of active locales.
     */
    public function updateNumActive()
    {
        $numActive = 0;
        $this->courses->each(function ($course) use (&$numActive) {
            $course->refresh(); // needs latest
            if ($course->isActive()) {
                $numActive++;
            }
        });
        if ($numActive != $this->num_active) {
            $this->num_active = $numActive;
            $this->save();
        }
    }
}
