<?php

namespace Ignite\Course\Models\Report;

use Ignite\Core\Entities\Filters\QueryPermissionFilters;
use Ignite\Core\Entities\User;
use Ignite\Core\Models\Grid\QueryTable;
use Ignite\Course\Entities\CourseCompletion;
use Illuminate\Support\Facades\DB;

class CourseFamilyReport extends QueryTable
{
    /**
     * The columns to show.
     *
     * @return array
     */
    public function columns()
    {
        $type = config('course.default_type');
        $columns = [
            'category' => [
                'title' => 'Category',
                'name' => 'activity_course_family.category',
            ],
            'family' => [
                'title' => __("Course::types.{$type}.Course Family"),
                'name' => 'activity_course_family.name as family',
            ],
            'num_completed' => [
                'title' => 'Completed',
                'name' => 'completions.num_completed',
            ],
        ];

        // add columns for each locale name
        $courseType = config('course.default_type');
        $locales = config("course.types.{$courseType}.locales");
        foreach ($locales as $locale => $localeName) {
            $columns['name_' . $locale] = [
                'title' => strtoupper($locale) . ' Name',
                'name' => 'course_' . $locale . '.name AS name_' . $locale,
            ];
        }

        foreach ($columns as $column => $value) {
            $columns[$column]['visible'] = true;
            $columns[$column]['orderable'] = true;
            $columns[$column]['exportable'] = true;
            $columns[$column]['searchable'] = true;
        }

        return $columns;
    }

    /**
     * The report query.
     *
     * @return \Illuminate\Database\Query\Builder
     */
    public function query()
    {
        $columns = $this->getColumnNames([])->toArray();

        // get the number of completions for each course
        $completions = CourseCompletion::query()
            ->select([
                'activity_course_completion.activity_course_family_id',
                DB::raw('COUNT(activity_course_completion.id) AS num_completed'),
            ])
            ->leftJoin('activity', 'activity_course_completion.activity_id', '=', 'activity.id')
            ->leftJoin('activity_submission', 'activity_submission.activity_id', '=', 'activity.id')
            ->leftJoin('core_user as user', 'activity_submission.user_id', '=', 'user.user_id')
            ->where('user.internal', User::TYPE_PARTICIPANT)
            ->groupBy('activity_course_completion.activity_course_family_id');

        $query = $this->getConnection()
            ->table('activity_course_family')
            ->select($columns)
            ->leftJoinSub($completions, 'completions', function ($join) {
                $join->on('activity_course_family.id', '=', 'completions.activity_course_family_id');
            });

        // add left join for each locale
        $courseType = config('course.default_type');
        $locales = config("course.types.{$courseType}.locales");
        foreach ($locales as $locale => $localeName) {
            $query->leftjoin('activity_course as course_' . $locale, function ($join) use ($locale) {
                $join->on('activity_course_family.id', '=', 'course_' . $locale . '.activity_course_family_id')
                    ->where('course_' . $locale . '.locale', $locale);
            });
        }

        return QueryPermissionFilters::for('core.user.participant.browse')
            ->apply($query, 'participant');
    }

    /**
     * Get a map of column keys => functions to format columns.
     *
     * @return array
     */
    protected function getColumnFormattingMap()
    {
        return [
            'num_completed' => function ($model) {
                return $model->num_completed ?? 0;
            },
        ];
    }
}
