<?php

namespace Ignite\Course\Jobs;

use Exception;
use Ignite\Course\Contracts\CourseRepositoryInterface;
use Ignite\Course\Entities\Course;
use Illuminate\Bus\Queueable;
use Illuminate\Console\Concerns\InteractsWithIO;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Symfony\Component\Console\Output\NullOutput;
use Symfony\Component\Console\Output\OutputInterface;
use ZipArchive;

class InitializeCourse implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithIO;
    use Queueable;
    use SerializesModels;

    /**
     * @var Course
     */
    protected Course $course;

    /**
     * @var boolean
     */
    protected bool $isDryRun = false;

    /**
     * @var boolean
     */
    protected bool $foundStorylineLmsJs = false;

    /**
     * @var boolean
     */
    protected bool $foundCompleteCourseJs = false;

    /**
     * @var boolean
     */
    protected bool $foundExitCourseJs = false;

    /**
     * @param Course  $course
     * @param boolean $isDryRun
     */
    public function __construct(Course $course, OutputInterface $output = null, $isDryRun = false)
    {
        $this->course = $course;
        $this->output = $output ?? new NullOutput();
        $this->isDryRun = $isDryRun;
    }

    /**
     */
    public function handle(): void
    {
        // create a lock so only one of this can run at a time
        $lockName = 'initialize_course_' . $this->course->id;
        DB::statement("SELECT GET_LOCK('{$lockName}', 300)");

        $meta = $this->course->meta;
        $uploaded = Storage::path($meta['uploadPathSource']);

        // set to pending first
        $meta['uploadStatus'] = Course::UPLOADSTATUS_PROCESSING;
        $meta['uploadErrors'] = null;
        $this->course->meta = $meta;
        $this->course->status = Course::PENDING;
        $this->course->save();

        try {
            $path = $this->unzipFile($uploaded, $meta['uploadNewCode']);
            $this->initialize($path);
            $this->moveCourseToTarget($path, $meta['uploadNewCode']);

            // update statutes
            $this->course->code = $meta['uploadNewCode'];
            $this->course->path = $this->course->code . '/story.html';
            $this->course->status = Course::ACTIVE;

            unset($meta['uploadPathSource']);
            unset($meta['uploadNewCode']);
            unset($meta['uploadStatus']);
            unset($meta['uploadErrors']);
            $this->course->meta = $meta;

            $this->course->save();

            $this->removeSource($uploaded);

            DB::statement("SELECT RELEASE_LOCK('{$lockName}')");
        } catch (Exception $e) {
            DB::statement("SELECT RELEASE_LOCK('{$lockName}')");

            $meta['uploadStatus'] = Course::UPLOADSTATUS_ERROR;
            $meta['uploadErrors'] = $e->getMessage();
            $this->course->meta = $meta;
            $this->course->save();
            $this->error($e->getMessage());
            logger()->warning($e);
        }
    }

    /**
     * Add the storyline-lms.js link.
     *
     * @param  string $content
     * @return string
     */
    protected function addStorylineLmsJs(string $content): string
    {
        $url = '/vendor/ignite-course/js/storyline-lms.js';
        $content = str_replace('</head>', "<script type='text/javascript' src='{$url}'></script>\n</head>", $content);

        return $content;
    }

    /**
     * @param  string $sourcePath
     * @param  string $target
     */
    protected function moveCourseToTarget(string $fullSourcePath, string $target)
    {
        $publicRoot = Storage::disk('public')->path('');
        $fullTargetPath = resolve(CourseRepositoryInterface::class)->getCoursePathBase() . '/' . $target;
        $targetPath = str_replace(Storage::disk('public')->path(''), '', $fullTargetPath);
        $sourcePath = str_replace(Storage::disk('local')->path(''), '', $fullSourcePath);

        if (strpos($fullTargetPath, $publicRoot) === false) {
            throw new Exception("The target path '{$fullTargetPath}' is not in the public folder.");
        }

        if (!is_dir($fullSourcePath)) {
            throw new Exception("The source path '{$fullSourcePath}' does not exist.");
        }

        if (!file_exists($fullSourcePath . '/story.html')) {
            throw new Exception("The source path '{$sourcePath}' does not contain story.html.");
        }

        if (is_dir($fullTargetPath)) {
            $this->warn("The target path '{$targetPath}' already exists. It will be replaced.");
            logger()->warning("The target path '{$targetPath}' already exists. It will be replaced.");
            if (!$this->isDryRun) {
                Storage::disk('public')->deleteDirectory($targetPath);
                if (is_dir($fullTargetPath)) {
                    throw new Exception("The old target path '{$fullTargetPath}' could not be deleted.");
                }
            }
        }

        if (!$this->isDryRun) {
            rename($fullSourcePath, $fullTargetPath);

            chmod($fullTargetPath, 0700);
            $files = Storage::disk('public')->allFiles($targetPath);
            foreach ($files as $filename) {
                chmod(Storage::disk('public')->path($filename), 0700);
            }
            $folders = Storage::disk('public')->allDirectories($targetPath);
            foreach ($folders as $folder) {
                chmod(Storage::disk('public')->path($folder), 0700);
            }
            if (!is_dir($fullTargetPath)) {
                throw new Exception("The target path '{$fullTargetPath}' still does not exist!");
            }
        }
    }

    /**
     * Find the expected content in the file.
     *
     * @param  string $content
     */
    protected function findExpectedContent(string $content)
    {
        if (preg_match('/\/storyline-lms.js/', $content)) {
            $this->foundStorylineLmsJs = true;
        }

        if (preg_match('/completeCourse()/', $content)) {
            $this->foundCompleteCourseJs = true;
        }
        if (preg_match('/exitCourse()/', $content)) {
            $this->foundExitCourseJs = true;
        }
    }

    /**
     * Fix mixed content of HTTP to HTTPS.
     *
     * @param  string $content
     * @return string
     */
    protected function fixMixedContent(string $content): string
    {
        $content = str_replace('http://ajax', '//ajax', $content);

        return $content;
    }

    /**
     * @param  string $fullpath
     */
    protected function initialize(string $fullpath)
    {
        $path = str_replace(Storage::path(''), '', $fullpath);

        $this->info('>> Replacing files in ' . $path . '...');
        if ($this->isDryRun) {
            $this->warn('This is a dry run.');
        }

        $this->foundStorylineLmsJs = false;
        $this->foundCompleteCourseJs = false;
        $this->foundExitCourseJs = false;

        $files = Storage::allFiles($path);
        foreach ($files as $file) {
            $content = Storage::get($file);
            $fname = str_replace($path . '/', '', $file);

            $content = $this->fixMixedContent($content);
            if ($file == $path . '/story.html') {
                $content = $this->addStorylineLmsJs($content);
            }

            $this->findExpectedContent($content);
            $this->warnsOfNewWindow($fname, $content);

            if (!$this->isDryRun) {
                Storage::put($file, $content);
            }
        }

        if (!$this->foundStorylineLmsJs) {
            $message = 'Could not inject JavaScript into story.html.';
            $message .= ' Please fix and re-upload.';
            throw new Exception($message);
        }
        if (!$this->foundCompleteCourseJs) {
            $message = 'No completeCourse() found in any button. Course may not be able to return the results.';
            $message .= ' Please fix and re-upload.';
            throw new Exception($message);
        }
        if (!$this->foundExitCourseJs) {
            $message = 'No exitCourse() found in any button. Course may not exit properly.';
            $message .= ' Please fix and re-upload.';
            throw new Exception($message);
        }
    }

    /**
     * @param  string $path
     */
    protected function removeSource(string $fullpath)
    {
        if (!$this->isDryRun) {
            if (file_exists($fullpath)) {
                $path = str_replace(Storage::path(''), '', $fullpath);
                Storage::delete($path);
                if (file_exists($fullpath)) {
                    throw new Exception("The source file '{$fullpath}' could not be deleted.");
                }
            }
        }
    }

    /**
     * Unzips the file.
     *
     * @param  string $filename
     * @param  string $code
     * @return string
     */
    protected function unzipFile(string $filename, string $code): string
    {
        $path = $code . uniqid();
        $fullpath = Storage::path($path);

        $this->info('>> Unzipping ' . basename($filename) . ' into ' . basename($path) . '...');
        $zip = new ZipArchive();
        $zip->open($filename);
        $zip->extractTo($fullpath);
        $zip->close();

        $files = Storage::files($path);
        $directories = Storage::directories($path);
        // if unzipped folder has only one subfolder, move it up
        if (count($files) === 0 && count($directories) === 1) {
            $newPath = $directories[0];
            $tempPath = $path . '_temp';
            Storage::move($newPath, $tempPath);
            Storage::deleteDirectory($path);
            Storage::move($tempPath, $path);
        }

        return $fullpath;
    }

    /**
     * Warns if the file opens a window in a new tab.
     *
     * @param  string $filename
     * @param  string $content
     */
    protected function warnsOfNewWindow(string $filename, string $content)
    {
        if (stripos($content, '"window":"_blank"') !== false) {
            $this->warn('File ' . $filename . ' has "window":"_blank"!');
            logger()->warning('File ' . $filename . ' has "window":"_blank"!');
        }
    }
}
