<?php

namespace Ignite\Course\Http\Forms;

use Ignite\Core\Models\Form\CoreForm;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use Kris\LaravelFormBuilder\Fields\FormField;

class CourseFamilyForm extends CoreForm
{
    /**
     * Build the enrollment form.
     *
     * @return void
     * @throws Exception
     */
    public function buildForm()
    {
        $this->setFormOptions([
            'id' => 'course-family-form',
        ]);

        $this->buildFromSchema(config("course.default_type"));

        if (isset($this->data['readonly'])) {
            /** @var FormField $field */
            foreach ($this->fields as $field) {
                $field->setOption('attr.readonly', true);
                $field->disable();

                if (in_array($field->getType(), ['button', 'submit', 'reset'])) {
                    $this->remove($field->getName());
                }
            }
        } else {
            $this->remove('num_active');
            $this->remove('created_at');
        }

        $this->fixDateField('publish_start');
        $this->fixDateField('publish_end');
    }

    /**
     * Fixes the date field by setting the correct date format instead of datetime.
     *
     * @param  string $field
     */
    protected function fixDateField(string $field)
    {
        $field = $this->getField($field);
        $value = $field->getValue();
        if (! $value) {
            return;
        }

        if (!is_string($value)) {
            $field->setOption('value', $value->format('Y-m-d'));
        }
    }

    /**
     * Get the schema.
     *
     * @param string $type
     * @return array
     */
    protected function getSchema(string $type)
    {
        $fields = config("course.types.{$type}.course_family_fields");
        $fields = array_merge(
            $fields,
            [
                'submit' => [
                    'frontend_type' => 'submit',
                    'name' => 'submit',
                    'label' => 'Submit',
                    'attr' => [
                        'id' => 'submit-course',
                        'class' => 'btn btn-primary pl-4 pr-4'
                    ],
                    'wrapper' => [
                        'class' => 'form-button text-right'
                    ],
                ],
            ],
        );

        // if thumbnail is not enabled, remove it from the schema
        $thumbnailEnabled = config("course.types.{$type}.course_family_fields.thumbnail.enable");
        if (!$thumbnailEnabled) {
            unset($fields['thumbnail']);
        }

        $schema = [
            'fields' => $fields,
        ];

        $this->schemaIndexFields($schema);
        return $this->schemaAlter($schema);
    }

    /**
     * Handle uploading files.
     * Overwrites the parent in Ignite\Core\Models\Form\Form.
     *
     * @param FormField $field
     * @param UploadedFile $file
     *
     * @return string
     */
    protected function handleFileUpload(FormField $field, UploadedFile $file)
    {
        if ($field->getOption('store_as')) {
            return parent::handleFileUpload($field, $file);
        }

        if ('thumbnail' !== $field->getName()) {
            return parent::handleFileUpload($field, $file);
        }

        $oldValue = $this->getModel()->thumbnail ?? null;
        $disk = $field->getOption('disk', 'public');
        $path = $field->getOption('path', 'courses/images');

        $filename = $file->getClientOriginalName();
        $ext = $file->getClientOriginalExtension();
        $basename = pathinfo($filename, PATHINFO_FILENAME);
        $name = $this->getField('name')->getValue() ?: $basename;
        $name = preg_replace('/[^A-Za-z0-9\-]/', '_', $name) . '_' . now()->format('YmdHis');

        // store new file
        $filename = $name . '.' . $ext;
        $file->storePubliclyAs($path, $filename, ['disk' => $disk]);

        // check new file exists
        if (!Storage::disk($disk)->exists($path . '/' . $filename)) {
            throw new \Exception("File upload failed for field {$field->getName()}");
        }

        // remove old file
        if ($oldValue && Storage::disk($disk)->exists($path . '/' . $oldValue)) {
            Storage::disk($disk)->delete($path . '/' . $oldValue);
        }

        return $filename;
    }
}
