<?php

namespace Ignite\Course\Models\Grid;

use Ignite\Core\Models\Grid\Formatter;
use Ignite\Core\Models\Grid\QueryTable;
use Ignite\Core\Models\Menu\ActionMenu;
use Ignite\Course\Entities\Course;
use Ignite\Course\Entities\CourseFamily;

class CourseFamilyTable extends QueryTable
{
    /**
     * @var Formatter
     */
    protected $formatter = null;

    /**
     * The columns to show.
     *
     * @return array
     */
    public function columns()
    {
        $type = config('course.default_type');
        $fields = config('course.types.' . $type . '.course_family_fields');

        return array_merge(
            ['actions' => $this->actionsColumn()],
            $fields
        );
    }

    /**
     * The participants query.
     *
     * @return \Illuminate\Database\Query\Builder
     */
    public function query()
    {
        $columns = $this->getColumnNames(['actions'])->toArray();
        $query = CourseFamily::getQuery()->select($columns);

        return $query;
    }

    /**
     * Custom column for actions.
     *
     * @param  Course $model
     * @return \Illuminate\Support\HtmlString
     */
    protected function actions($model)
    {
        $type = config('course.default_type');
        $manager = resolve(ActionMenu::class);
        $manager->merge(
            $manager->hydrateItems(
                collect([
                    [
                        'position' => 10,
                        'url' => route(
                            'admin.course_family.show',
                            ['type' => $type, 'courseFamily' => $model->id]
                        ),
                        'label' => 'View',
                        'icon' => 'eye-open',
                        'allow' => auth()->user()->can('activity.course.all.view'),
                        'children' => [
                            [
                                'position' => 10,
                                'url' => route(
                                    'admin.course_family.edit',
                                    ['type' => $type, 'courseFamily' => $model->id]
                                ),
                                'label' => 'Edit',
                                'icon' => 'edit',
                                'allow' => auth()->user()->can('activity.course.all.update'),
                            ],
                            [
                                'position' => 20,
                                'url' => route(
                                    'admin.course_family.history',
                                    ['type' => $type, 'courseFamily' => $model->id]
                                ),
                                'label' => 'History',
                                'icon' => 'history',
                                'allow' => auth()->user()->can('activity.course.all.view'),
                            ],
                            [
                                'position' => 30,
                                'url' => route(
                                    'admin.course.index',
                                    ['type' => $type, 'courseFamily' => $model->id]
                                ),
                                'label' => 'Trainings',
                                'icon' => 'table',
                                'allow' => auth()->user()->can('activity.course.all.view'),
                            ],
                        ]
                    ],
                ])
            )
        );

        return $this->formatter()->html(
            $manager->render([
                'primaryKey' => $model->id,
                'model' => CourseFamily::class
            ])
        );
    }

    /**
     * @return GenericFormatter
     */
    protected function formatter(): Formatter
    {
        if (null === $this->formatter) {
            $this->formatter = resolve(Formatter::class);
        }

        return $this->formatter;
    }

    /**
     * Get a map of column keys => functions to format columns.
     *
     * @return array
     */
    protected function getColumnFormattingMap()
    {
        return [
            'actions' => function ($model) {
                return $this->actions($model);
            },
        ];
    }

    /**
     * Get default builder parameters.
     *
     * @return array
     */
    protected function getBuilderParameters1()
    {
        $params = [
            'order' => [
                [2, 'asc'],
            ],
        ];

        return parent::getBuilderParameters(
            $params,
            ...func_get_args()
        );
    }
}
