<?php

namespace Ignite\Course\Providers;

use Ignite\Core\Models\Source\Manager;
use Ignite\Course\Contracts\CourseRepositoryInterface;
use Ignite\Course\Providers\EventServiceProvider;
use Ignite\Course\Providers\RouteServiceProvider;
use Ignite\Course\Providers\ViewServiceProvider;
use Ignite\Course\Repositories\CourseRepository;
use Illuminate\Support\Facades\App;
use Illuminate\Support\ServiceProvider;

class CourseServiceProvider extends ServiceProvider
{
    /**
     * @var string
     */
    protected $path = null;

    /**
     * All of the container bindings that should be registered.
     *
     * @var array
     */
    public $bindings = [];

    /**
     * All of the container singletons that should be registered.
     *
     * @var array
     */
    public $singletons = [
        CourseRepositoryInterface::class => CourseRepository::class,
    ];

    /**
     * The available commands
     *
     * @var array
     */
    protected $commands = [
        \Ignite\Course\Console\Initialize::class,
    ];

    /**
     * Boot the application events.
     *
     * @return void
     */
    public function boot()
    {
        $this->bootFactories();
        $this->bootMigrations();
        $this->bootTranslations();
        $this->bootHelper();
        $this->bootViews();
        $this->bootAssets();
    }

    /**
     * Register the service provider.
     *
     * @return void
     */
    public function register()
    {
        $this->app->register(RouteServiceProvider::class);
        $this->app->register(EventServiceProvider::class);
        $this->app->register(ViewServiceProvider::class);

        $this->registerConfig();
        $this->registerCommands();
        $this->registerActivityTypeStrategy();
        $this->registerSourceModels();
    }

    /**
     * @return void
     */
    protected function bootAssets()
    {
        $sourcePath = $this->getpath('/resources/assets');
        $publishPath = public_path('vendor/ignite-course');
        $this->publishes([$sourcePath => $publishPath], ['assets', "ignite-course-assets"]);
    }

    /**
     * Register an additional directory of factories.
     *
     * @return void
     */
    protected function bootFactories()
    {
        if (! app()->environment('production') && $this->app->runningInConsole()) {
            $this->loadFactoriesFrom($this->getpath('/database/factories'));
        }
    }

    /**
     * @return void
     */
    protected function bootHelper()
    {
        require_once $this->getpath('/src/helpers.php');
    }

    /**
     * Boot the directory of migrations.
     *
     * @return void
     */
    protected function bootMigrations()
    {
        $programPath = module_path('Program', '/database/migrations');
        if (is_dir($programPath)) {
            $this->publishes([
                $this->getpath('/database/migrations') => $programPath,
            ], "ignite-course-migration");
        } else {
            $this->publishes([
                $this->getpath('/database/migrations') => base_path('database/migrations'),
            ], "ignite-course-migration");
        }

        if (config('course.use_default_migration')) {
            $this->loadMigrationsFrom([
                $this->getpath('/database/migrations'),
            ]);
        }
    }

    /**
     * Register translations.
     *
     * @return void
     */
    protected function bootTranslations()
    {
        $langPath1 = $this->getpath('/resources/lang');
        $langPath2 = resource_path('lang/vendor/ignite-course');

        $this->loadTranslationsFrom($langPath1, 'Course');

        if (is_dir($langPath2)) {
            $this->loadTranslationsFrom($langPath2, 'Course');
        }

        $this->publishes([$langPath1 => $langPath2], ['translations', "ignite-course-translations"]);
    }

    /**
     * Register views.
     *
     * @return void
     */
    protected function bootViews()
    {
        $sourcePath = $this->getpath('/resources/views');
        $publishPath = resource_path("views/vendor/ignite-course");

        $this->publishes([$sourcePath => $publishPath], ['views', "ignite-course-views"]);

        $viewPaths = [];
        foreach ($this->app->config->get('view.paths') as $path) {
            if (is_dir($path . '/vendor/ignite-course')) {
                $viewPaths[] = $path . '/vendor/ignite-course';
            }
        }
        if (is_dir($publishPath)) {
            $viewPaths[] = $publishPath;
        }
        $viewPaths[] = $sourcePath;

        $this->loadViewsFrom(array_unique($viewPaths), 'Course');
    }

    /**
     * @param  string $path
     * @return string
     */
    protected function getPath($path): string
    {
        if (null === $this->path) {
            $this->path = __DIR__ . '/../../';
        }

        return realpath($this->path . $path);
    }

    /**
     * Registers the activity type strategy for the course types.
     */
    protected function registerActivityTypeStrategy()
    {
        $types = config('course.types');
        foreach ($types as $type => $config) {
            $this->app->singleton(
                'ActivityStrategy' . ucwords($type),
                function () use ($config, $type) {
                    return resolve(
                        $config['activityTypeStrategy'],
                        ['type' => $type]
                    );
                }
            );
        }
    }

    /**
     * Register the commands.
     */
    protected function registerCommands()
    {
        if (App::runningInConsole()) {
            $this->commands($this->commands);
        }
    }

    /**
     * Register config.
     *
     * @return void
     */
    protected function registerConfig()
    {
        $this->publishes([
            $this->getpath('/config/course.php') => config_path('course.php'),
        ], "ignite-course-config");

        $this->mergeConfigFrom(
            $this->getpath('/config/course.php'),
            'course'
        );
    }

    /**
     * Register the source models.
     *
     * @return void
     */
    protected function registerSourceModels()
    {
        $sources = array_keys(config('course.source.models'));
        $sourceManager = resolve(Manager::class);
        foreach ($sources as $source) {
            $sourceManager->register($source, config("course.source.models.{$source}"));
        }
    }
}
