<?php

namespace Ignite\Course\Http\Forms;

use Ignite\Core\Models\Form\CoreForm;
use Ignite\Course\Entities\Course;
use Illuminate\Http\UploadedFile;
use Kris\LaravelFormBuilder\Fields\FormField;

class CourseForm extends CoreForm
{
    /**
     * Build the enrollment form.
     *
     * @return void
     * @throws Exception
     */
    public function buildForm()
    {
        $this->setFormOptions([
            'id' => 'course-form',
        ]);

        $this->buildFromSchema(config("course.default_type"));

        if (isset($this->data['readonly'])) {
            $this->remove('upload');

            /** @var FormField $field */
            foreach ($this->fields as $field) {
                $field->setOption('attr.readonly', true);
                $field->disable();

                if (in_array($field->getType(), ['button', 'submit', 'reset'])) {
                    $this->remove($field->getName());
                }
            }
        } else {
            $this->setStatusOptions();
            $this->remove('path');
            $this->remove('created_at');

            if (!$this->getModel()) {
                $this->remove('status');
            }
        }

        $this->setCountriesCheckedValues();
    }

    /**
     * Get the schema.
     *
     * @param string $type
     * @return array
     */
    protected function getSchema(string $type)
    {
        $fields = config("course.types.{$type}.course_fields");
        $fields = array_merge(
            $fields,
            [
                'submit' => [
                    'frontend_type' => 'submit',
                    'name' => 'submit',
                    'label' => 'Submit',
                    'attr' => [
                        'id' => 'submit-course',
                        'class' => 'btn btn-primary pl-4 pr-4'
                    ],
                    'wrapper' => [
                        'class' => 'form-button text-right'
                    ],
                ],
            ],
        );

        $schema = [
            'fields' => $fields,
        ];

        $this->schemaIndexFields($schema);
        return $this->schemaAlter($schema);
    }

    /**
     * Handle uploading files.
     * Overwrites the parent in Ignite\Core\Models\Form\Form.
     *
     * @param FormField $field
     * @param UploadedFile $file
     *
     * @return string
     */
    protected function handleFileUpload(FormField $field, UploadedFile $file)
    {
        return $file->store('uploads', 'local');
    }

    /**
     */
    protected function setCountriesCheckedValues()
    {
        $field = $this->getField('countries');
        $value = $field->getValue();
        if (! $value) {
            return;
        }
        if (!is_array($value)) {
            $values = explode(',', $value);
        } else {
            $values = $value;
        }

        foreach ($field->getChildren() as $child) {
            $child->setOption('checked', in_array($child->getOption('value'), $values));
        }
    }

    /**
     * Set the status options.
     */
    protected function setStatusOptions()
    {
        $field = $this->getField('status');
        $choices = $field->getOption('choices');
        $current = $field->getValue();

        if ($this->getModel()) {
            $choices = array_filter($choices, function ($key) use ($current) {
                if ($current == Course::PENDING) {
                    // if pending, then only pending
                    return $key == Course::PENDING;
                } else {
                    // if not pending, then do not show pending
                    return $key != Course::PENDING;
                }
            }, ARRAY_FILTER_USE_KEY);
        } else {
            // if new, then only pending
            $choices = array_filter($choices, function ($key) {
                return $key == Course::PENDING;
            }, ARRAY_FILTER_USE_KEY);
        }
        $field->setOption('choices', $choices);
    }
}
