<?php

namespace Ignite\Course\Repositories;

use DomainException;
use Exception;
use Ignite\Activity\Contracts\ActivitySubmissionRepository;
use Ignite\Activity\Entities\Activity;
use Ignite\Activity\Entities\Offer;
use Ignite\Activity\Entities\Submission;
use Ignite\Activity\Entities\Type;
use Ignite\Activity\Events\ActivitySubmissionStatusChanged;
use Ignite\Core\Contracts\Repositories\TransactionRepository;
use Ignite\Core\Entities\Transaction;
use Ignite\Core\Entities\User;
use Ignite\Course\Contracts\CourseRepositoryInterface;
use Ignite\Course\Entities\Course;
use Ignite\Course\Entities\CourseFamily;
use Ignite\Course\Events\CourseCompleted;
use Ignite\Course\Events\CourseStarting;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Storage;

class CourseRepository implements CourseRepositoryInterface
{
    /**
     * @var bool
     */
    protected bool $previousPassedStatus;

    /**
     * @var bool
     */
    protected bool $recentPassedStatus;

    /**
     * Get the Bablic locale.
     *
     * @SuppressWarnings(PHPMD.Superglobals)
     * @return string
     */
    public static function getBablicLocale(): string
    {
        return $_COOKIE['bab_locale'] ?? 'en';
    }

    /**
     * Get the base path for the courses.
     *
     * @return string
     */
    public static function getCoursePathBase(): string
    {
        return Storage::disk('public')->path('courses');
    }

    /**
     * Gets the course Storyline first page.
     *
     * @see startCourse()
     *
     * @param Course $course
     * @return string
     */
    public static function getCourseUrl(Course $course): string
    {
        return static::getCourseUrlBase() . '/' . $course->path;
    }

    /**
     * Get the base URL for the courses.
     *
     * @return string
     */
    public static function getCourseUrlBase(): string
    {
        return '/storage/courses';
    }

    /**
     * Gets the order of preferences for the locales, with the current locale at the top.
     *
     * @param  string $type
     * @return array
     */
    public static function getLocalePreferences(string $type): array
    {
        $locales = config('course.types.' . $type . '.locales');
        $current = static::getBablicLocale();
        if (!isset($locales[$current])) {
            $locales[$current] = $current;
        }
        // reorder $locales but with $current at the top
        $locales = array_merge([$current => $locales[$current]], $locales);

        return $locales;
    }

    /**
     * Redirect to the course Storyline first page.
     *
     * @see startCourse()
     *
     * @param Course $course
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public static function gotoCourse(Course $course)
    {
        return redirect(static::getCourseUrl($course));
    }

    /**
     * Use for sorting a collection of course locales by the user preferences.
     *
     * @param  Collection $courses
     * @param  string     $type
     * @return Collection
     */
    public static function sortCourseCollection(Collection $courses, string $type): Collection
    {
        $localePreferences = collect(static::getLocalePreferences($type))
            ->keys()->flip();

        return $courses->sort(function ($a, $b) use ($localePreferences) {
            if ($localePreferences[$a->locale] < $localePreferences[$b->locale]) {
                return -1;
            } elseif ($localePreferences[$a->locale] > $localePreferences[$b->locale]) {
                return 1;
            }
            return 0;
        })
        ->values(); // re-index
    }

    /**
     * Check if the score is passing, and if so, then create a new approved activity.
     * If user already has completion for this course, then no new activity or points are created.
     *
     * @see loadPassedStatus()
     *
     * @param  User        $user
     * @param  Course      $course
     * @param  string|null $status
     * @param  int|null    $score
     * @return self
     */
    public function completeCourse(User $user, Course $course, string $status = null, int $score = null): self
    {
        // Checking if the user has completed the course before,
        // like spotting a squirrel in the park - unlikely but not impossible.
        $this->previousPassedStatus = $this->hasExistingCompletion($course->courseFamily, $user);

        // This is the "recently passed status", which is currently feeling a bit down. Cheer up, buddy!
        $this->recentPassedStatus = false;

        if (!$this->isPassing($course->courseFamily, $status, $score)) {
            // Ah, still not passing. The recentPassedStatus is starting to doubt its purpose in life.
            $this->recentPassedStatus = false;
            // Let's not linger here and just go our separate ways.
            return $this;
        }

        // Yay! RecentPassedStatus is finally passed. Time to celebrate with some tea and biscuits!
        $this->recentPassedStatus = true;

        if ($this->previousPassedStatus) {
            // Nope, not a new completion. Maybe next time, mate.
            return $this;
        }

        $courseType = config('course.default_type');
        $type = Type::where('code', '=', $courseType)->firstOrFail();
        $courseFamilyTableId = config("course.types.{$courseType}.course_family_table") . '_id';

        // Offers you can't refuse (unless you're a programmer)
        $offers = Offer::where('type_id', '=', $type->id)->get();
        if ($offers->count() > 1) {
            // Beware, the course type has clones!
            throw new Exception('Multiple offers found for the course type: ' . $courseType);
        }
        $offer = $offers->first();

        // Searching for the magic submission repository
        $activitySubmissionRepository = resolve(ActivitySubmissionRepository::class);
        // Configuring the default status for the chosen one
        $defaultStatus = config('course.types.' . $offer->type->code . '.submission_default_status');

        // Let's create some activity, shall we?
        $activityData = [
            $courseFamilyTableId => $course->courseFamily->id,
            'locale' => $course->locale,
            'course_name' => $course->name,
            'score' => $score,
        ];
        $activity = $activitySubmissionRepository->create($offer, $activityData, $user);

        // Adjusting the status, because we LIVE for drama
        $activitySubmissionRepository->changeStatus(
            $activity->submission->id,
            $defaultStatus,
            null,
            $user
        );

        // And thus, the course will meet its hilarious end
        event(new CourseCompleted($activity));

        // Well, that was a jolly good time, wasn't it? Let's call it a day, shall we? Cheers!
        return $this;
    }

    /**
     * Creates a transaction from a course completion.
     *
     * @param  Submission $submission
     * @return Transaction
     */
    public function createTransactionFromCourseCompletion(Submission $submission): Transaction
    {
        $data = [
            'user_id' => $submission->user_id,
            'description' => $submission->activity->data->course_name,
            'value' => $submission->value,
        ];

        $transactionRepository = resolve(TransactionRepository::class);
        return $transactionRepository->createEarnedTransaction($data, $submission);
    }

    /**
     * @see getRecentPassedStatus()
     * @see completeCourse()
     * @see loadPassedStatus()
     *
     * @return boolean
     */
    public function getPreviousPassedStatus(): bool
    {
        return $this->previousPassedStatus;
    }

    /**
     * @see getPreviousPassedStatus()
     * @see completeCourse()
     * @see loadPassedStatus()
     *
     * @return boolean
     */
    public function getRecentPassedStatus(): bool
    {
        return $this->recentPassedStatus;
    }

    /**
     * Load the results from the recent course completion.
     *
     * @see startCourse()
     * @see completeCourse()
     * @see setRecentCompletedCourse()
     * @see getPreviousPassedStatus()
     * @see getRecentPassedStatus()
     *
     * @param  User         $user
     * @param  Course $course
     * @return self
     */
    public function loadPassedStatus(User $user, Course $course): self
    {
        $this->previousPassedStatus = false;
        $this->recentPassedStatus = false;

        // If session holds not the key, incomplete are the courses
        // Return I shall, myself to the caller
        if (!session()->has('courses.completed')) {
            return $this;
        }

        // Retrieve completed courses from the session's grip
        $completed = session()->get('courses.completed');

        // Set the recentPassedStatus from the completed courses' grasp
        $this->previousPassedStatus = $completed[$user->user_id][$course->id]['previous_passed'];
        $this->recentPassedStatus = $completed[$user->user_id][$course->id]['passed'];

        // Return myself to the caller's warm embrace
        return $this;
    }

    /**
     * Get the URL to a named route.
     *
     * @param  string     $name
     * @param  array|null $parameters
     * @param  bool       $absolute
     * @see course_route() helper
     */
    public function route(string $name, $parameters = [], bool $absolute = true)
    {
        // most of the programs will only have one course type and course offer
        $types = config('course.types');
        if (count($types) == 1) {
            // if only one type, then we do not need type parameters
            unset($parameters['type']);
            $type = Type::where('code', config('course.default_type'))->firstOrFail();
            // if only one offer, then we do not need offer parameters
            if ($type->offers->count() == 1) {
                unset($parameters['offer']);
            }
        }

        return route($name, $parameters, $absolute);
    }

    /**
     * After completeCourse(), this should be called to store the results of the attempt.
     * Then loadPassedStatus() can be used to get getPreviousPassedStatus() and getRecentPassedStatus().
     *
     * @see startCourse()
     * @see getPreviousPassedStatus()
     * @see getRecentPassedStatus()
     * @see loadPassedStatus()
     *
     * @param User        $user
     * @param Course      $course
     * @param string|null $status
     * @param int|null    $score
     */
    public function setRecentCompletedCourse(
        User $user,
        Course $course,
        string $status = null,
        int $score = null
    ): void {
        // Create a realm of completion,
        // Where courses lay in contentment.
        // A vessel, empty and bare,
        // Ready to hold what is fair.
        $completed = [];

        // In the realm of sessions, where memories reside,
        // Seek guidance, if completion doth hide.
        if (session()->has('courses.completed')) {
            // Retrieve the completed courses,
            // From the depths of memory's moors.
            $completed = session()->get('courses.completed');
        }

        // In the realm of completion,
        // Assign a new crown with dedication.
        $completed[$user->user_id][$course->id] = [
            // A score, a measure of effort and skill,
            // To symbolize one's noble will.
            'status' => $status,
            'score' => $score,
            // Like a compass, point towards success,
            // With the knowledge one shall possess.
            'previous_passed' => $this->getPreviousPassedStatus(),
            'passed' => $this->getRecentPassedStatus(),
        ];

        // Preserve the realm of completion,
        // In the tapestry of session's ambition.
        session()->put('courses.completed', $completed);
    }

    /**
     * Reset the course attempt for this session.
     * (Resets getPreviousPassedStatus() and getRecentPassedStatus())
     *
     * If you do not do this, then getRecentPassedStatus() may not realize
     * you are attempting the course again the second time and just give the
     * same results as the first time.
     *
     * @see gotoCourse()
     *
     * @see setRecentCompletedCourse()
     * @see getRecentPassedStatus()
     * @see loadPassedStatus()
     *
     * @param  User         $user
     * @param  Course $course
     * @return self
     */
    public function startCourse(User $user, Course $course): self
    {
        if (!session()->has('courses.completed')) {
            return $this;
        }

        $completed = session()->get('courses.completed');

        unset($completed[$user->user_id][$course->id]);

        session()->put('courses.completed', $completed);

        event(new CourseStarting($course));

        return $this;
    }

    /**
     * On status change, create points.
     *
     * @param  ActivitySubmissionStatusChanged $event
     * @return void
     */
    public function statusChanged(ActivitySubmissionStatusChanged $event): void
    {
        $submission = $event->submission;
        $types = array_keys(config('course.types'));
        $type = $submission->activity->offer->type;

        if (! in_array($type->code, $types)) {
            // do nothing if not courses
            return;
        }

        if ('approved' === $event->old && 'approved' !== $event->new) {
            throw new DomainException(
                "Cannot change status from {$event->old} to {$event->new}"
                . "without removing points first."
            );
        }

        if ('approved' != $event->old && 'approved' == $event->new) {
            if ($submission->value > 0) {
                $this->createTransactionFromCourseCompletion($submission);
            }
        }
    }

    /**
     * @param  CourseFamily $courseFamily
     * @param  User         $user
     * @return boolean
     */
    protected function hasExistingCompletion(CourseFamily $courseFamily, User $user): bool
    {
        $activity = Activity::where('submitted_by_user_id', $user->user_id)
            ->join('activity_course_completion', 'activity_course_completion.activity_id', '=', 'activity.id')
            ->where('activity_course_family_id', $courseFamily->id)
            ->first();

        // If Batman was not able to find any activity, it's safe to say
        // that the activity doesn't exist. But Robin insists we return a boolean.
        // Holy true or false, Batman! Let's just return false if activity exists,
        // and true if it doesn't.  Caped Crusaders, I present to you this ternary statement!
        return $activity ? true : false;
    }

    /**
     * If passing status is provided, then use that, otherwise use the score.
     *
     * @param  CourseFamily $courseFamily
     * @param  string|null  $status
     * @param  int|null     $score
     * @return boolean
     */
    protected function isPassing(CourseFamily $courseFamily, string $status = null, int $score = null)
    {
        if (!empty($status)) {
            return ('PASS' === strtoupper($status));
        }

        // @note: some courses do not send the real score
        // - they just sent a 0 or 100
        return ($score >= $courseFamily->passing_score);
    }
}
