<?php

namespace Ignite\Course\Http\Controllers;

use Exception;
use Ignite\Activity\Entities\Activity;
use Ignite\Activity\Entities\Offer;
use Ignite\Activity\Entities\Submission;
use Ignite\Activity\Entities\Type;
use Ignite\Core\Http\Controllers\Controller;
use Ignite\Course\Contracts\CourseRepositoryInterface;
use Ignite\Course\Entities\Course;
use Ignite\Course\Events\CourseEnded;
use Ignite\Course\Events\CourseExited;
use Illuminate\Http\Request;

class CourseController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @param Request     $request
     * @param string|null $category
     * @return Renderable|RedirectResponse|JsonResponse
     */
    public function index(Request $request, string $category = null)
    {
        $type = $this->getRequestType($request);
        $offer = $this->getRequestOffer($request, $type);

        return view()->first([
            "Course::{$type->code}.index",
            "Course::course.index",
        ], compact('type', 'offer', 'category'));
    }

    /**
     * This should be posted from a JS file with the course completion score.
     * Some courses post to this page first, but some may go directly to courseEnd() instead.
     *
     * @param  Request $request
     * @param  CourseRepositoryInterface $courseRepository
     * @return JsonResponse
     */
    public function complete(Request $request, CourseRepositoryInterface $courseRepository)
    {
        $user = auth()->user();
        // $title = $request->get('title');
        // $points = $request->get('points');
        $percent = $request->get('percent');
        $status = $request->get('status');

        $path = $request->get('path');
        $referer = $request->headers->get('referer');
        // $referer = Request::header('referer');
        // $referer = Request::server('HTTP_REFERER');

        // $title = ($title === 'null') ? null : $title;
        $percent = ($percent === 'null') ? null : $percent;
        $status = ($status === 'null') ? null : $status;
        $path = ($path === 'null') ? null : $path;

        $course = $this->findCourseInfo($path ? $path : $referer);
        if (!$course) {
            throw new Exception("Unknown course from {$path} / {$referer}");
        }

        $score = $percent;
        if (!is_numeric($score)) {
            if (!empty($status)) {
                $score = 'PASS' === strtoupper($status) ? 100 : 0;
            } else {
                $score = 0;
            }
        }

        $courseRepository->completeCourse($user, $course, $status, $score);

        // Store something to session so next page (end())
        // will know what their new score is, and not think it is being
        // completed a second time.
        $courseRepository->setRecentCompletedCourse($user, $course, $status, $score);

        return response()->json(['success' => true]);
    }

    /**
     * Display a listing of the completed courses.
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function completed()
    {
        // set the activity type for the results
        $type = Type::where('code', config('course.default_type'))->first();
        Activity::setDefaultType($type->code);

        $submissions = Submission::query()
            ->select('activity_submission.*')
            ->with(['activity', 'activity.data', 'state'])
            ->leftJoin('activity', 'activity.id', '=', 'activity_submission.activity_id')
            ->leftJoin('activity_offer', 'activity_offer.id', '=', 'activity.offer_id')
            ->where('activity_submission.user_id', auth()->id())
            ->where('activity_offer.type_id', $type->id)
            ->get();

        return view('Course::course.completed', compact('type', 'submissions'));
    }

    /**
     * @param Request $request
     * @param CourseRepositoryInterface $courseRepository
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function end(Request $request, CourseRepositoryInterface $courseRepository)
    {
        $user = auth()->user();
        $path = $request->get('path');
        $referer = $request->headers->get('referer');
        $course = $this->findCourseInfo($path ? $path : $referer);

        if (!$course) {
            throw new Exception("Unknown course from {$path} / {$referer}");
        }

        event(new CourseEnded($course));

        // They may have already posted the course completion score on the "previous" page
        // where the new points were calculated, and now is just the post-summary redirect.
        $courseRepository->loadPassedStatus($user, $course);

        $previousPassed = $courseRepository->getPreviousPassedStatus();
        $hasPassed = $courseRepository->getRecentPassedStatus();

        if ($previousPassed && $hasPassed) {
            $this->flashSuccess(__('Course::types.course.course_passed_again', ['course' => $course->name]));
            return redirect(route('activity.course.completed'));
        } elseif (!$previousPassed && $hasPassed) {
            $this->flashSuccess(__('Course::types.course.course_passed', ['course' => $course->name]));
            return redirect(route('activity.course.completed'));
        } else {
            $this->flashInfo(__('Course::types.course.course_not_passed', ['course' => $course->name]));
            return redirect(
                course_route(
                    'activity.course.index',
                    [
                        'type' => $course->courseFamily->offer->type->code,
                        'offer' => $course->courseFamily->offer->code,
                    ]
                )
            );
        }
    }

    /**
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function exit(Request $request)
    {
        $path = $request->get('path');
        $referer = $request->headers->get('referer');
        $course = $this->findCourseInfo($path ? $path : $referer);

        event(new CourseExited($course));

        if (!$course) {
            return redirect(course_route('activity.course.index'));
        }

        return redirect(
            course_route(
                'activity.course.index',
                [
                    'type' => $course->courseFamily->offer->type->code,
                    'offer' => $course->courseFamily->offer->code,
                ]
            )
        );
    }

    /**
     * @param Course              $course
     * @param CourseRepositoryInterface $courseRepository
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function start(Course $course, CourseRepositoryInterface $courseRepository)
    {
        // If you do not do startCourse(), then getRecentPassedStatus()
        // may not realize you are attempting the course again the
        // second time and just give the same results as the first time.
        $courseRepository->startCourse(auth()->user(), $course);

        return $courseRepository->gotoCourse($course);
    }


    /**
     * @param  string|null $path
     * @return Course|null
     */
    protected function findCourseInfo(string $path = null): ?Course
    {
        return Course::onlyActive()->findByUrl($path)->first();
    }

    /**
     * @param  Request $request
     * @param  Type    $type
     * @return Offer
     */
    protected function getRequestOffer(Request $request, Type $type): Offer
    {
        $offer = $request->get('offer');
        if (null !== $offer) {
            return $type->offers()->where('code', $offer)->firstOrFail();
        } else {
            return $type->getFirstOffer();
        }
    }

    /**
     * @param  Request $request
     * @return Type    $type
     */
    protected function getRequestType(Request $request): Type
    {
        $type = $request->get('type', collect(config('course.types'))->keys()->first());
        return Type::where('code', $type)->firstOrFail();
    }
}
