<?php

namespace Ignite\Course\Http\ViewComposers;

use Ignite\Activity\Entities\Offer;
use Ignite\Core\Entities\User;
use Ignite\Course\Contracts\CourseRepositoryInterface;
use Ignite\Course\Entities\CourseFamily;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;
use Illuminate\View\View;

class CourseListComposer
{
    /**
     * @var array
     */
    protected array $config;

    /**
     * Bind data to the view.
     *
     * @param  View  $view
     * @return void
     */
    public function compose(View $view)
    {
        $user = auth()->check() ? auth()->user() : null;
        $type = $view->offsetGet('type');
        $this->config = config('course.types.' . $type->code);

        $offer = $view->offsetGet('offer');
        $category = $view->offsetGet('category');
        $categories = $this->config['categories'];

        if ($category !== null || empty($categories)) {
            $this->composeCourses($view, $offer, $user, $category);
        } else {
            $this->composeCategories($view, $user, $offer);
        }
    }

    /**
     * @param  View   $view
     * @param  array  $config
     */
    protected function composeCategories(View $view, User $user, Offer $offer)
    {
        $categoryList = $this->config['categories'];
        $images = $this->config['category_images'];
        $courseFamilies = $this->getCourseFamilies(
            $user,
            $offer,
            $user ? $user->participant->country : $this->getDefaultCountry()
        )
            ->filter(function ($courseFamily) {
                return $this->isCourseActiveOrCompleted($courseFamily);
            });

        $categories = [];
        foreach ($categoryList as $key => $category) {
            $categories[$key] = (object) [
                'name' => $category,
                'image' => $images[$key],
                'numCourses' => $courseFamilies->filter(function ($courseFamily) use ($key) {
                    return $courseFamily->category === $key;
                })->count(),
                'numCompletions' => $courseFamilies->filter(function ($courseFamily) use ($key) {
                    return ($courseFamily->category === $key && $courseFamily->completed == 1);
                })->count(),
            ];
        }

        $view->with('categories', $categories);
        $view->with('hideEmptyCategories', $this->config['hideEmptyCategories']);
    }

    /**
     * Compose the courses and completions for the view of courses in a category.
     *
     * @param  View   $view
     * @param  Offer  $offer
     * @param  User   $user
     * @param  string $category
     */
    protected function composeCourses(View $view, Offer $offer, User $user, string $category = null)
    {
        $courseFamilyTableId = $this->config['course_family_table'] . '_id';

        // get the completions for the user, group by course,
        // and get the first completion in the list
        $completions = $this->getCourseCompletions($user)
            ->groupBy($courseFamilyTableId)
            ->map(function ($completions) {
                return $completions->first();
            });

        $courseFamilies = $this->getCourseFamilies(
            $user,
            $offer,
            $user ? $user->participant->country : $this->getDefaultCountry()
        )
            // filter by category
            ->filter(function ($courseFamily) use ($category) {
                return $category === null || $courseFamily->category === $category;
            })
            // only show active or completed course families
            ->filter(function ($courseFamily) {
                return $this->isCourseActiveOrCompleted($courseFamily);
            })
            ->map(function ($courseFamily) use ($offer) {
                // sort the courses in order of current locale
                // @phpcs:disable Generic.Files.LineLength.TooLong
                $courseFamily->courses = resolve(CourseRepositoryInterface::class)::sortCourseCollection($courseFamily->courses, $offer->type->code);
                // @phpcs:enable
                return $courseFamily;
            });

        $view
            ->with('courseFamilies', $courseFamilies)
            ->with('completions', $completions);
    }

    /**
     * Get the completions for the user order by course, score, and created_at.
     *
     * @param  User|null $user
     * @return Collection
     */
    protected function getCourseCompletions(User $user = null): Collection
    {
        if (!$user) {
            return collect();
        }

        $courseFamilyTableId = $this->config['course_family_table'] . '_id';
        $completionTable = $this->config['completion_table'];
        $completionModel = $this->config['completion_model'];

        return $completionModel::query()
            ->select("{$completionTable}.*")
            ->join('activity', $completionTable . '.activity_id', '=', 'activity.id')
            ->join('activity_submission', 'activity.id', '=', 'activity_submission.activity_id')
            ->where('activity_submission.user_id', $user->user_id)
            ->orderBy($courseFamilyTableId, 'asc')
            ->orderBy('score', 'asc')
            ->orderBy('created_at', 'asc')
            ->get();
    }

    /**
     * Get the courses possibly available for the user sorted.
     *
     * @param  User|null   $user
     * @param  Offer       $offer
     * @param  string|null $country
     * @return Collection
     */
    protected function getCourseFamilies(User $user = null, Offer $offer, string $country = null): Collection
    {
        $courseFamilyModel = $this->config['course_family_model'];
        $courseFamilyTable = $this->config['course_family_table'];
        $courseFamilyTableId = $this->config['course_family_table'] . '_id';
        $completionTable = $this->config['completion_table'];
        $completionModel = $this->config['completion_model'];

        $completionSubquery = $completionModel::query()
            ->select([
                $courseFamilyTableId,
                DB::raw("IF(MIN({$completionTable}.id) > 0, 1, 0) AS completed"),
            ])
            ->leftJoin('activity', 'activity.id', '=', 'activity_id')
            ->leftJoin('activity_submission', 'activity.id', '=', 'activity_submission.activity_id')
            ->where('activity_submission.user_id', $user->user_id ?? 0)
            ->groupBy($courseFamilyTableId);

        $query = $courseFamilyModel::query()
            ->with('courses')
            ->select(["{$courseFamilyTable}.*", 'completed', ])
            ->leftJoinSub(
                $completionSubquery,
                'completion',
                function ($join) use ($courseFamilyTable, $courseFamilyTableId) {
                    $join->on("{$courseFamilyTable}.id", '=', "completion.{$courseFamilyTableId}");
                }
            )
            ->forOffer($offer)
            ->orderBy('category', 'asc')
            ->orderBy('sequence', 'desc')
            ->orderBy('publish_start', 'desc')
            ->orderBy('created_at', 'desc');

        if ($country) {
            $query->whereHas('locales', function ($query) use ($country) {
                $query->forCountry($country);
            });
        }

        return $query->get();
    }

    /**
     * @return string|null
     */
    protected function getDefaultCountry(): ?string
    {
        return null;
    }

    /**
     * Is the course currently active or completed by user.
     *
     * @param  CourseFamily $courseFamily
     * @return boolean
     */
    protected function isCourseActiveOrCompleted(CourseFamily $courseFamily): bool
    {
        if ($courseFamily->isActive() && $courseFamily->isCurrentlyPublsh()) {
            return true;
        }
        if ($courseFamily->completed) {
            return true;
        }

        return false;
    }
}
