<?php

namespace Ignite\Course\Entities;

use Ignite\Core\Entities\Base;
use Ignite\Course\Entities\CourseFamily;
use Ignite\Course\Jobs\InitializeCourse;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Http\UploadedFile;
use Illuminate\Support\Facades\Storage;
use OwenIt\Auditing\Auditable;
use OwenIt\Auditing\Contracts\Auditable as AuditableContract;

/**
 * Ignite\Course\Entities\Course
 *
 * @property int $id
 * @property int|null $activity_course_family_id
 * @property string|null $name
 * @property string|null $description
 * @property string|null $locale
 * @property string|null $countries
 * @property string|null $path
 * @property string $status
 * @property string|null $code
 * @property array|null $meta
 * @property \Illuminate\Support\Carbon|null $deleted_at
 * @property \Illuminate\Support\Carbon|null $created_at
 * @property \Illuminate\Support\Carbon|null $updated_at
 * @property-read \Illuminate\Database\Eloquent\Collection|\Ignite\Core\Entities\Audit[] $audits
 * @property-read int|null $audits_count
 * @property-read CourseFamily|null $courseFamily
 * @method static Builder|Course forCountry(string $country)
 * @method static Builder|Course newModelQuery()
 * @method static Builder|Course newQuery()
 * @method static Builder|Course onlyActive()
 * @method static Builder|Course onlyForLocale($locale)
 * @method static \Illuminate\Database\Query\Builder|Course onlyTrashed()
 * @method static Builder|Course query()
 * @method static Builder|Course whereActivityCourseFamilyId($value)
 * @method static Builder|Course whereCode($value)
 * @method static Builder|Course whereCountries($value)
 * @method static Builder|Course whereCreatedAt($value)
 * @method static Builder|Course whereDeletedAt($value)
 * @method static Builder|Course whereDescription($value)
 * @method static Builder|Base whereHasPermission(string $permission, ?\Ignite\Core\Entities\User $user = null)
 * @method static Builder|Course whereId($value)
 * @method static Builder|Course whereLocale($value)
 * @method static Builder|Course whereMeta($value)
 * @method static Builder|Course whereName($value)
 * @method static Builder|Course wherePath($value)
 * @method static Builder|Course whereStatus($value)
 * @method static Builder|Course whereUpdatedAt($value)
 * @method static \Illuminate\Database\Query\Builder|Course withTrashed()
 * @method static \Illuminate\Database\Query\Builder|Course withoutTrashed()
 * @mixin \Eloquent
 */
class Course extends Base implements AuditableContract
{
    use Auditable;
    use SoftDeletes;

    public const PENDING = 'pending';
    public const INACTIVE = 'inactive';
    public const ACTIVE = 'active';

    public const UPLOADSTATUS_PENDING = 'pending';
    public const UPLOADSTATUS_PROCESSING = 'processing';
    public const UPLOADSTATUS_ERROR = 'error';

    /**
     * The table name.
     * @var string
     */
    protected $table = 'activity_course';

    /**
     * Should the timestamps be audited?
     *
     * @var bool
     */
    protected $auditTimestamps = false;

    /**
     * Custom Audit Driver
     *
     * @var \Ignite\Core\Audit\ParticipantDriver
     */
    protected $auditDriver = \Ignite\Core\Audit\GenericDriver::class;

    /**
     * Attributes to exclude from the Audit.
     *
     * @var array
     */
    protected $auditExclude = [];

    /**
     * The attributes that should be casted.
     *
     * @var array
     */
    protected $casts = [
        'active' => 'boolean',
        'meta' => 'json',
        'deleted_at' => 'datetime',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * @var string
     */
    protected $courseType = 'course';

    /**
     * The identifier for the record in the audit log.
     *
     * @return mixed
     */
    public static function getAuditFriendlyField()
    {
        return ['code', 'name'];
    }

    /**
     * Get the status list.
     *
     * @return array
     */
    public static function getStatusList(): array
    {
        return [
            self::PENDING => 'Pending',
            self::ACTIVE => 'Active',
            self::INACTIVE => 'Inactive',
        ];
    }

    /**
     * The relationship to the CourseFamily model.
     *
     * @return BelongsTo
     */
    public function courseFamily(): BelongsTo
    {
        return $this->belongsTo(CourseFamily::class, (new CourseFamily())->getTable() . '_id');
    }

    /**
     * Get the status of the upload.
     *
     * @return string
     */
    public function getPathStatus(): string
    {
        return $this->meta['uploadStatus'] ?? '';
    }

    /**
     * Use meta to get the status of the files at path.
     *
     * @return string
     */
    public function getPathStatusMessage(): string
    {
        if (!empty($this->meta['uploadErrors'])) {
            if (is_array($this->meta['uploadErrors'])) {
                return implode(' ', $this->meta['uploadErrors']);
            } else {
                return $this->meta['uploadErrors'];
            }
        }

        if (isset($this->meta['uploadStatus']) && self::UPLOADSTATUS_PENDING == $this->meta['uploadStatus']) {
            return __("Course::types.{$this->courseType}.message.upload_status.pending");
        }

        if (isset($this->meta['uploadStatus']) && self::UPLOADSTATUS_PROCESSING == $this->meta['uploadStatus']) {
            return __("Course::types.{$this->courseType}.message.upload_status.processing");
        }

        return '';
    }

    /**
     * Returns if the course is active.
     *
     * @return boolean
     */
    public function isActive(): bool
    {
        return $this->status == self::ACTIVE;
    }

    /**
     * Returns if the course is inactive.
     *
     * @return boolean
     */
    public function isInactive(): bool
    {
        return $this->status == self::INACTIVE;
    }

    /**
     * Returns if the course is pending
     *
     * @return boolean
     */
    public function isPending(): bool
    {
        return $this->status == self::PENDING;
    }

    /**
     * Scope a query to include only active records.
     *
     * @param Builder $query
     * @param string  $country
     */
    public function scopeForCountry(Builder $query, string $country): void
    {
        $query->where('countries', 'LIKE', '%' . $country . '%');
    }

    /**
     * Only active courses.
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     */
    public function scopeOnlyActive($query): void
    {
        $query->where('status', self::ACTIVE);
    }

    /**
     * Only course available in certain locale
     *
     * @param \Illuminate\Database\Eloquent\Builder $query
     */
    public function scopeOnlyForLocale($query, $locale): void
    {
        $query->where('locale', '=', $locale);
    }

    /**
     * Update the path of the course.
     *
     * @param UploadedFile $file
     */
    public function updatePath(string $path)
    {
        if (Storage::exists($path)) {
            // make sure no other course has this code
            $newCode = str_slug($this->name, '-');
            $existingCourse = Course::where('code', $newCode)
                ->where('id', '!=', $this->id)
                ->first();
            if ($existingCourse) {
                $newCode .= '-' . $this->id;
            }

            $meta = $this->meta;
            $meta['uploadPathSource'] = $path;
            $meta['uploadNewCode'] = $newCode;
            $meta['uploadStatus'] = self::UPLOADSTATUS_PENDING;
            $meta['uploadErrors'] = null;
            $this->meta = $meta;
            $this->save();

            dispatch(new InitializeCourse($this));
        }
    }

    /**
     * The url to the offer page on the front-end.
     * @return string
     */
    public function url(): string
    {
        return course_route('activity.course.start', [
            // 'type' => 'course',
            'course' => $this->code,
        ]);
    }

    /**
     * Boots some events for model.
     */
    protected static function boot()
    {
        parent::boot();

        static::saving(function ($model) {
            if (!$model->code) {
                $model->code = str_slug($model->name, '_');
            }
        });

        static::saved(function ($model) {
            if ($model->isDirty('status')) {
                $model->courseFamily->updateNumActive();
            }
        });
    }
}
