<?php

namespace Ignite\Core\Http\Controllers\Admin;

use Ignite\Core\Contracts\Repositories\GroupRepository;
use Ignite\Core\Contracts\Repositories\UserRepository;
use Ignite\Core\Entities\User;
use Ignite\Core\Http\Controllers\Controller;
use Ignite\Core\Models\Grid\UserTable;
use Illuminate\Http\Request;

class UsersController extends Controller
{
    /**
     * @var UserRepository
     */
    private $userRepository;
    /**
     * @var GroupRepository
     */
    private $groupRepository;

    /**
     * UsersController constructor.
     *
     * @param UserRepository $userRepository
     * @param GroupRepository $groupRepository
     */
    public function __construct(UserRepository $userRepository, GroupRepository $groupRepository)
    {
        $this->userRepository = $userRepository;
        $this->groupRepository = $groupRepository;
    }

    /**
     * List the users in the system.
     *
     * @param  UserTable $table
     * @param  Request   $request
     * @return \Illuminate\Http\JsonResponse|\Illuminate\View\View
     */
    public function index(UserTable $table, Request $request)
    {
        if ($request->wantsJson()) {
            return $table->ajax();
        }

        return $table->render('Core::admin.security.users.index');
    }

    /**
     * Display a form to edit the relationship between a user and their associated groups.
     *
     * @param  int $user
     * @return \Illuminate\View\View
     */
    public function edit($user)
    {
        $user = $this->userRepository->find($user);
        $groups = $this->groupRepository->getAccessibleGroups();

        return view('Core::admin.security.users.edit', compact('user', 'groups'));
    }

    /**
     * Update an existing group.
     *
     * @param  Request $request
     * @return \Illuminate\Http\RedirectResponse|string
     * @throws \Illuminate\Validation\ValidationException
     */
    public function update(Request $request)
    {
        $this->validate($request, [
            'user_id' => 'exists:core_user,user_id',
            'groups' => 'required|array'
        ]);

        try {
            /** @var User $user */
            $user = $this->userRepository->assignGroups($request->get('user_id'), $request->get('groups'));
            $this->flashSuccess('User group assignment updated successfully.');
            return redirect()->route('admin.security.users.edit', $user);
        } catch (\Exception $e) {
            $this->logException($e);
            $this->flashError('User group assignment was not updated.');
            return redirect()->route('admin.security.users.edit', $user);
        }
    }

    /**
     * Activate a user.
     *
     * @param  int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function activate($id)
    {
        try {
            $this->userRepository->activate($id);
            $this->flashSuccess('User was activated successfully.');
        } catch (\Exception $e) {
            $this->logException($e);
            $this->flashError('User could not be activated.');
        }

        return redirect()->route('admin.security.users.index');
    }

    /**
     * Deactivate a user.
     *
     * @param  int $id
     * @return \Illuminate\Http\RedirectResponse
     */
    public function deactivate($id)
    {
        try {
            $this->userRepository->deactivate($id);
            $this->flashSuccess('User was deactivated successfully.');
        } catch (\Exception $e) {
            $this->logException($e);
            $this->flashError('User could not be deactivated.');
        }

        return redirect()->route('admin.security.users.index');
    }
}
