<?php

namespace Ignite\Core\Models\Grid;

use Ignite\Core\Entities\Permission;
use Illuminate\Database\DatabaseManager;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Collection;
use Yajra\DataTables\DataTables;

class PermissionTable extends EloquentTable
{
    /**
     * @var PermissionFormatter
     */
    protected PermissionFormatter $formatter;

    /**
     * PermissionTable constructor.
     *
     * @param DataTables $datatables
     * @param DatabaseManager $databaseManager
     * @param PermissionFormatter $formatter
     * @param array $params
     */
    public function __construct(
        DataTables $datatables,
        DatabaseManager $databaseManager,
        PermissionFormatter $formatter,
        array $params = []
    ) {
        parent::__construct($datatables, $databaseManager, $params);

        $this->formatter = $formatter;
    }

    /**
     * The columns to show.
     *
     * @return array
     */
    public function columns(): array
    {
        return [
            'checkbox' => [
                'defaultContent' => '',
                'title'          => '',
                'data'           => 'checkbox',
                'name'           => 'checkbox',
                'orderable'      => false,
                'searchable'     => false,
                'exportable'     => false,
                'printable'      => false,
                'width'          => '10px',
            ],
            'action' => [
                'name' => 'actions',
                'title' => 'Actions',
                'searchable' => false,
                'orderable' => false,
                'exportable' => false,
                'printable' => false,
                'visible' => true,
                'width' => '80px',
                'class' => 'actions'
            ],
            'id' => [
                'title' => 'Permission ID',
                'name' => 'id',
                'orderable' => true,
                'exportable' => true,
            ],
            'key' => [
                'title' => 'Key',
                'name' => 'key',
                'orderable' => true,
                'exportable' => true,
            ],
            'description' => [
                'title' => 'Description',
                'name' => 'description',
                'orderable' => true,
                'exportable' => true,
            ],
            'status' => [
                'title' => 'Status',
                'name' => 'status',
                'orderable' => true,
                'exportable' => true,
            ],
            'is_user_defined' => [
                'title' => 'User Defined',
                'name' => 'is_user_defined',
                'orderable' => true,
                'exportable' => true,
            ],
        ];
    }

    /**
     * The permission query.
     *
     * @return \Illuminate\Database\Eloquent\Builder|\Illuminate\Database\Query\Builder|Collection
     */
    public function query(): \Illuminate\Database\Eloquent\Builder|\Illuminate\Database\Query\Builder|Collection
    {
        return Permission::query();
    }

    /**
     * Get a map of column keys => functions to format columns.
     *
     * @return array
     */
    protected function getColumnFormattingMap(): array
    {
        return [
            'checkbox' => function ($model) { return $this->formatter->checkbox($model); },
            'action' => function ($model) { return $this->formatter->actions($model); },
            'status' => function ($model) { return $this->formatter->status($model); },
            'is_user_defined' => function ($model) { return $this->formatter->isUserDefined($model); },
        ];
    }

    /**
     * Get default builder parameters.
     *
     * @return array
     */
    protected function getBuilderParameters(): array
    {
        return parent::getBuilderParameters([
            'order' => [[2, 'asc']], // Permission ID
            'select' => [
                'info' => true,
                'style' => 'multi',
                'selector' => 'td:first-child'
            ],
            'columnDefs' => [
                ['targets' => 0, 'checkboxes' => ['selectRow' => true]]
            ]
        ]);
    }
}
