<?php

namespace Ignite\Core\Models\Grid;

use Ignite\Core\Entities\Participant;
use Illuminate\Database\DatabaseManager;
use Ignite\Core\Entities\Audit;
use Illuminate\Database\Eloquent\Builder;
use Yajra\DataTables\DataTables;

class ParticipantHistoryTable extends EloquentTable
{
    /**
     * The excluded column keys.
     * @var array
     */
    protected $excludedKeys = [];

    /**
     * The column keys that should be always visible.
     * @var array
     */
    protected $alwaysVisible = [];

    /**
     * @var ParticipantFormatter
     */
    protected $formatter;

    /**
     * ParticipantTable constructor.
     *
     * @param DataTables       $datatables
     * @param DatabaseManager  $databaseManager
     * @param HistoryFormatter $formatter
     * @param array            $params
     */
    public function __construct(
        DataTables $datatables,
        DatabaseManager $databaseManager,
        HistoryFormatter $formatter,
        array $params = []
    ) {
        parent::__construct($datatables, $databaseManager, $params);

        $this->formatter = $formatter;
    }

    /**
     * The columns to show.
     *
     * @return array
     */
    public function columns(): array
    {
        return [
            'created_at' => [
                'title' => 'Created',
                'name' => 'created_at',
                'orderable' => true,
                'exportable' => true
            ],
            'user_id' => [
                'title' => 'Changed By',
                'name' => 'user_id',
                'orderable' => true,
                'exportable' => true,
            ],
            'event' => [
                'title' => 'Event',
                'name' => 'event',
                'orderable' => true,
                'exportable' => true,
            ],
            'old_values' => [
                'title' => 'Message',
                'name' => 'old_values',
                'orderable' => true,
                'exportable' => true,
            ],
        ];
    }

    /**
     * The participant history query.
     *
     * @return Builder
     */
    public function query(): Builder
    {
        return Audit::with('user')
            ->where('context_type', Participant::class)
            ->where('context_id', $this->user_id);
    }

    /**
     * Get a map of column keys => functions to format columns.
     *
     * @return array
     */
    protected function getColumnFormattingMap(): array
    {
        return [
            'created_at' => function ($model) { return $this->formatter->createdAt($model); },
            'user_id' => function ($model) { return $this->formatter->user($model); },
            'event' => function ($model) { return $this->formatter->event($model); },
            'old_values' => function ($model) { return $this->formatter->message($model); },
        ];
    }

    /**
     * Get default builder parameters.
     *
     * @return array
     */
    protected function getBuilderParameters(): array
    {
        return parent::getBuilderParameters([
            'order' => [
                [2, 'desc'] // Event
            ]
        ]);
    }
}
