<?php

namespace Ignite\Core\Models\Menu;

use Ignite\Core\Contracts\CanDisplayInMenu;
use Ignite\Core\Events\Menus\NavigationMenuPreCompile;
use Illuminate\Support\Str;
use Spatie\Menu\Laravel\Link;
use Spatie\Menu\Laravel\Menu as SpatieMenu;

class NavigationMenu extends Manager
{
    /** @var SpatieMenu */
    protected $menu;

    /**
     * Add a link to the menu.
     *
     * @param CanDisplayInMenu $item
     * @param SpatieMenu $menu
     */
    public function addLink(CanDisplayInMenu|array $item, $menu)
    {
        $link = $menu->linkIf($item->canBeViewed(), $item->getUrl(), $this->buildLabel($item));
        $link->addItemParentClass('page-nav-item');
        $link->setActiveClass('page-nav-item--active');
        $link->setActive(function ($link) use ($item, $menu) {
            /** @var \Spatie\Menu\Link $link */
            if ($link->url() === url()->current()) {
                return true;
            }

            $isNotHomepage = url($link->url()) !== url('/');
            $containsUrl = Str::contains(url()->current(), $link->url());

            if ($isNotHomepage && $containsUrl && ! $menu->isExactActive()) {
                return true;
            }

            return false;
        });
    }

    /**
     * Build the label for the link,
     *
     * @param  CanDisplayInMenu $item
     * @return string
     */
    protected function buildLabel(CanDisplayInMenu $item)
    {
        return $item->getLabel();
    }

    /**
     * Build a submenu link.
     *
     * @param  CanDisplayInMenu $item
     * @return Link
     */
    public function addSubmenuLink(CanDisplayInMenu $item)
    {
        return Link::to('#' . $item->getUrl(), sprintf('%s <span class="caret"></span>', $item->getLabel()))
            ->addClass('dropdown-toggle')
            ->setAttribute('data-toggle', 'dropdown')
            ->setAttribute('aria-haspopup', 'true')
            ->setAttribute('aria-expanded', 'false');
    }

    /**
     * Build the menu.
     *
     * @param  \Illuminate\Support\Collection|Iterator|array $links
     * @return SpatieMenu
     */
    public function buildMenu($links): SpatieMenu
    {
        return MenuExtended::build($links->toArray(), function ($menu, $item) {
            if (is_array($item)) {
                $item = new Item($item['url'], $item['label'], $item['position'], $item['allow'], $item['children'], $item['icon'], $item['badge']);
            }

            if (! $item->hasChildren()) {
                $this->addLink($item, $menu);
            } else {
                $menu->submenu(
                    $this->addSubmenuLink($item),
                    $this->buildMenu($item->getChildren())
                         ->addClass('dropdown-menu page-nav-dropdown')
                         ->addParentClass('page-nav-item page-nav-item--has-dropdown dropdown')
                );
            }
        });
    }

    /**
     * Render the menu.
     */
    public function render(): string
    {
        event(new NavigationMenuPreCompile($this));

        $this->menu = $this->buildMenu($this->getItems())->addClass('page-nav-list page-nav--right');

        return $this->menu->render();
    }
}
