<?php

namespace Ignite\Core\Models\Menu;

use Ignite\Core\Contracts\CanDisplayInMenu;
use Ignite\Core\Events\Menus\AdminMenuPreCompile;
use Illuminate\Support\Str;
use Spatie\Menu\Laravel\Facades\Menu;
use Spatie\Menu\Laravel\Html;
use Spatie\Menu\Laravel\Link;
use Spatie\Menu\Laravel\Menu as SpatieMenu;

class AdminMenu extends Manager
{
    /** @var SpatieMenu */
    private $menu;

    /**
     * Add a link to the menu.
     *
     * @param CanDisplayInMenu $item
     * @param SpatieMenu $menu
     */
    public function addLink(CanDisplayInMenu $item, $menu)
    {
        $menu->addItemParentClass('nav-item');
        $menu->addItemClass('nav-link');
        $link = $menu->link($item->getUrl(), $this->buildLabel($item));
        $link->setActive(function ($link) use ($item) {
            if (route('homepage').$link->url() === url()->current()) {
                return true;
            }

            if ($link->url() === url()->current()) {
                return true;
            }

            if (Str::contains(url()->current(), $link->url()) && url($link->url()) !== url('admin')) {
                return true;
            }

            return false;
        });
    }

    /**
     * Build the label for the link,
     *
     * @param  CanDisplayInMenu $item
     * @return string
     */
    protected function buildLabel(CanDisplayInMenu $item)
    {
        return Html::raw(sprintf(
            '<i class="nav-icon fas fa-%s"></i><p>%s</p>',
            $item->getIcon() ?? 'circle',
            $item->getLabel()
        ))->html();
    }

    /**
     * Build a submenu link.
     *
     * @param  CanDisplayInMenu $item
     * @return Link
     */
    public function addSubmenuLink(CanDisplayInMenu $item)
    {
        $child = sprintf(
            '<i class="nav-icon fas fa-%s"></i><p>%s</p> <i class="right fas fa-angle-left"></i>',
            $item->getIcon(),
            $item->getLabel()
        );

        return Link::to('#', $child);
    }

    /**
     * Build the menu.
     *
     * @param  \Illuminate\Support\Collection|array $links
     * @return SpatieMenu
     */
    public function buildMenu($links)
    {
        return Menu::build($links->toArray(), function ($menu, $item) {
            if (is_array($item)) {
                $item = new Item($item['url'], $item['label'], $item['position'], $item['allow'], $item['children'], $item['icon'], $item['badge']);
            }


            if (! $item->hasChildren()) {
                $this->addLink($item, $menu);
            } else {
                $menu->submenu(
                    $this->addSubmenuLink($item, $menu)
                        ->addClass('nav-link'),
                    $this->buildMenu($item->getChildren())
                         ->addClass('nav nav-treeview')
                         ->addParentClass('has-treeview')
                );
            }
        });
    }

    /**
     * Render the menu.
     *
     * @return string
     */
    public function render()
    {
        event(new AdminMenuPreCompile($this));

        $this->menu = $this->buildMenu($this->getItems())
            ->addClass('nav nav-pills nav-sidebar flex-column')
            ->setAttribute('data-widget', 'treeview')
            ->setAttribute('role', 'menu')
        ;

        return $this->menu->render();
    }
}
