<?php

namespace Ignite\Core\Models\Grid;

use Ignite\Core\Entities\Filters\QueryPermissionFilters;
use Ignite\Core\Entities\TransactionResource;
use Illuminate\Database\DatabaseManager;
use Illuminate\Database\Eloquent\Builder;
use Yajra\DataTables\DataTables;

class WaitingTransactionsTable extends EloquentTable
{
    /**
     * The excluded column keys.
     *
     * @var array
     */
    protected $excludedKeys = [];

    /**
     * The column keys that should be always visible.
     *
     * @var array
     */
    protected $alwaysVisible = ['identifier', 'value'];

    /**
     * @var WaitingTransactionsFormatter
     */
    protected $formatter;

    /**
     * ParticipantTable constructor.
     *
     * @param DataTables $datatables
     * @param DatabaseManager $databaseManager
     * @param WaitingTransactionsFormatter $formatter
     * @param array $params
     */
    public function __construct(
        DataTables $datatables,
        DatabaseManager $databaseManager,
        WaitingTransactionsFormatter $formatter,
        array $params = []
    )
    {
        parent::__construct($datatables, $databaseManager, $params);

        $this->formatter = $formatter;
    }

    /**
     * The columns to show.
     *
     * @return array
     */
    public function columns(): array
    {
        return [
            'checkbox' => [
                'defaultContent' => '',
                'title' => '',
                'data' => 'checkbox',
                'name' => 'checkbox',
                'orderable' => false,
                'searchable' => false,
                'exportable' => false,
                'printable' => false,
                'width' => '10px',
            ],
            'action' => [
                'name' => 'action',
                'title' => 'Actions',
                'searchable' => false,
                'orderable' => false,
                'exportable' => false,
                'printable' => false,
                'visible' => true,
                'width' => '140px',
                'class' => 'actions',
            ],
            'id' => [
                'title' => 'ID',
                'name' => 'id',
                'orderable' => true,
                'exportable' => true,
            ],
            'identifier' => [
                'title' => 'Identifier',
                'name' => 'identifier',
                'orderable' => true,
                'exportable' => true,
            ],
            'type' => [
                'title' => 'Type',
                'name' => 'type',
                'orderable' => true,
                'exportable' => true,
            ],
            'description' => [
                'title' => 'Description',
                'name' => 'description',
                'orderable' => true,
                'exportable' => true,
            ],
            'tax_date' => [
                'title' => 'Tax Date',
                'name' => 'tax_date',
                'orderable' => true,
                'exportable' => true,
            ],
            'transaction_date' => [
                'title' => 'Transaction Date',
                'name' => 'transaction_date',
                'orderable' => true,
                'exportable' => true,
            ],
            'value' => [
                'title' => 'Value',
                'name' => 'value',
                'orderable' => true,
                'exportable' => true,
            ]
        ];
    }

    /**
     * The waiting transactions query.
     *
     * @return Builder
     */
    public function query(): Builder
    {
        $columns = $this->getColumnNames(['action', 'checkbox'])->toArray();

        return QueryPermissionFilters::for('core.transactions.waiting.browse')->apply(
            TransactionResource::query()
                ->select($columns)
        );
    }

    /**
     * Get a map of column keys => functions to format columns.
     *
     * @return array
     */
    protected function getColumnFormattingMap(): array
    {
        return [
            'action' => [$this->formatter, 'actions'],
            'value' => [$this->formatter, 'balance'],
            'checkbox' => [$this->formatter, 'checkbox']
        ];
    }

    /**
     * Get default builder parameters.
     *
     * @return array
     */
    protected function getBuilderParameters(): array
    {
        $columns = array_flip(array_keys($this->columns()));
        $sortBy = $columns['transaction_date'] ?? ($columns['id'] ?? 2);

        return parent::getBuilderParameters([
            'order' => [
                [$sortBy, 'desc']
            ],
            'select' => [
                'info' => true,
                'style' => 'multi',
                'selector' => 'td:first-child'
            ],
            'columnDefs' => [
                ['targets' => 0, 'checkboxes' => ['selectRow' => true]]
            ]
        ]);
    }
}
