<?php

namespace Ignite\Core\Models\Grid;

use Ignite\Core\Entities\Participant;
use Ignite\Core\Entities\User;
use Ignite\Core\Models\Menu\ActionMenu;
use Ignite\Core\Program;
use Illuminate\Support\HtmlString;

class ParticipantFormatter extends Formatter
{
    /**
     * @var Program
     */
    protected Program $program;

    /**
     */
    protected static array|string|null $userStatusList = null;

    /**
     * ParticipantFormatter constructor.
     *
     * @param Program $program
     */
    public function __construct(Program $program)
    {
        $this->program = $program;
    }

    /**
     * Custom column for actions.
     *
     * @param  object  $model
     * @return HtmlString
     */
    public function actions(object $model): HtmlString
    {
        /** @var ActionMenu $manager */
        $manager = resolve(ActionMenu::class);
        $manager->merge(
            $manager->hydrateItems(
                collect([
                    [
                        'position' => 10,
                        'url' => route('admin.participant.show', $model->user_id),
                        'label' => 'View',
                        'icon' => 'eye-open',
                        'allow' => auth()->user()->can('core.user.participant.browse'),
                        'children' => [
                            [
                                'position' => 20,
                                'url' => route('admin.participant.edit', $model->user_id),
                                'label' => 'Edit',
                                'icon' => 'edit',
                                'allow' => auth()->user()->can('core.user.participant.update')
                            ], [
                                'position' => 50,
                                'url' => route('admin.participant.transactions', [$model->user_id]),
                                'label' => 'View Transactions',
                                'icon' => 'exchange',
                                'allow' => auth()->user()->can('core.user.transaction.browse')
                            ], [
                                'position' => 60,
                                'url' => route('admin.participant.history', [$model->user_id]),
                                'label' => 'View History',
                                'icon' => 'history',
                                'allow' => auth()->user()->can('core.user.history.browse')
                            ], [
                                'position' => 70,
                                'url' => route('admin.participant.activity', [$model->user_id]),
                                'label' => 'View Activity',
                                'icon' => 'heartbeat',
                                'allow' => auth()->user()->can('core.user.activity.browse')
                            ], [
                                'position' => 80,
                                'url' => route('admin.participant.notes', [$model->user_id]),
                                'label' => 'View Notes',
                                'icon' => 'sticky-note',
                                'allow' => auth()->user()->can('core.user.activity.browse')
                            ], [
                                'position' => 90,
                                'url' => route('admin.impersonate', $model->user_id),
                                'label' => 'Impersonate',
                                'icon' => 'user-secret',
                                'allow' => auth()->user()->can('core.user.impersonate')
                            ]
                        ]
                    ],
                ])
            )
        );

        return $this->html($manager->render([
            'primaryKey' => $model->user_id,
            'model' => Participant::class
        ]));
    }

    /**
     * Format the participant's full name.
     *
     * @param  object $model
     * @return string
     */
    public function name(object $model): string
    {
        if ($model instanceof Participant) {
            return $model->fullName();
        }

        return "{$model->first} {$model->last}";
    }

    /**
     * Format the agree confirm 1 as Yes or No.
     *
     * @param  object $model
     * @return string
     */
    public function agreeConfirm1(object $model): string
    {
        return $this->formatYesNo($model->agree_confirm_1);
    }

    /**
     * Format agree confirm 2 as Yes or No.
     *
     * @param  object $model
     * @return string
     */
    public function agreeConfirm2(object $model): string
    {
        return $this->formatYesNo($model->agree_confirm_2);
    }

    /**
     * Format internal as Yes or No.
     *
     * @param  object $model
     * @return string
     */
    public function internal(object $model): string
    {
        return $this->formatYesNo($model->internal);
    }

    /**
     * Format archived as Yes or No.
     *
     * @param  object $model
     * @return string
     */
    public function archived(object $model): string
    {
        return $this->formatYesNo($model->archived);
    }

    /**
     * Get a participant checkbox based on their ID.
     *
     * @param  object $model
     * @return HtmlString
     */
    public function checkbox(object $model): HtmlString
    {
        return $this->formatCheckbox($model->user_id);
    }

    /**
     * Format the last login at column.
     *
     * @param  object $model
     * @return HtmlString
     */
    public function lastLoginAt(object $model): HtmlString
    {
        if (empty($model->last_login_at)) {
            return $this->html(__('Never'));
        }

        return $this->formatDate($model, 'last_login_at', 'Y-m-d H:i:s');
    }

    /**
     * Format the enrollment date column.
     *
     * @param  object $model
     * @return HtmlString
     */
    public function enrolledAt(object $model): HtmlString
    {
        return $this->formatDate($model, 'created_at', 'Y-m-d H:i:s');
    }

    /**
     * Format the enrollment date column.
     *
     * @param  object  $model
     * @return HtmlString
     */
    public function status(object $model): HtmlString
    {
        if (null == static::$userStatusList) {
            static::$userStatusList = resolve(User::class)->getStatusList();
        }
        return static::$userStatusList[$model->status] ?? $model->status;
    }
}
